;**********************************************************************
;                                                                     *
;   NOM:     fcounter14.asm                                           *
;   Date:    19/12/2011                                               *
;   Version: 1.4                                                      *
;   Circuit: Frquencemtre                                           *
;   Auteur:  Bernard DECAESTECKER F8EOZ									  *
;	Documentation: Sandeep Guria (Ham India), Peter Ouwehand, Loc    *
;   Lefebvre, Peter Cousens, Sprut                                                                  *
;**********************************************************************
;                                                                     *
;    Fichier requis: P16F84.inc                                       *
;                                                                     *
;**********************************************************************
;                                                                     *
;   LCD JHD 162A : 2 * 16  											  *
;   with HD44780U dot-matrix liquid crystal display controller and 	  *
;   driver LSI displays alphanumerics.                                *
;**********************************************************************
;*  PIC pin
;*	
;*	PORTA:
;*	 	RA0 Setup 
;*		RA1	Setup next 
;*		RA3 Control frequency input
;*		RA4 Frequency input
;*	PORTB:	0 LCD Display E
;*		RB1=RS
;*		RB2=RW
;*		RB3=E ; 	
;*		RB7-RB4 = DB7-DB4  
;*	
;*****************************************************************************
;                             DEFINES                         
;*****************************************************************************
;
;	Choose our PIC: 16F84 or other
;	Comment out the next line [;#define F84] if using other PIC
#define	F84
#ifndef	F84
	#define	Fother
#endif

;	Choose our XTAL: 4 MHz or other
;	Comment out the next line [;#define XTAL4] if using other XTAL
#define	XTAL4	; XTAL = 4 MHz
#ifndef	XTAL4
	#define	XTAL10	; XTAL = 10 MHz
#endif
;-----------------------------------------------------------------------------
;	CPU configuration
;
#ifdef	F84
	MESSG		"Processor = 16F84"
	#define 	RAMStart	0x0C	; 68 General Purpose registers (SRAM)
	#define 	EEPROMStart	0x2100	; EEPROM
	processor	16f84				; Dfinitions des constantes
	include		<p16f84.inc>
	__CONFIG   _CP_OFF & _WDT_OFF & _PWRTE_ON & _HS_OSC
; '__CONFIG' prcise les paramtres encods dans le processeur au moment de
; la programmation. Les dfinitions sont dans le fichier include.
; Voici les valeurs et leurs dfinitions :
;	_CP_ON				Code protection ON : impossible de relire
;	_CP_OFF				Code protection OFF
;	_PWRTE_ON			Timer reset sur power on en service
;	_PWRTE_OFF			Timer reset hors-service
;	_WDT_ON				Watch-dog en service
;	_WDT_OFF			Watch-dog hors service
;	_LP_OSC				Oscillateur quartz basse vitesse
;	_XT_OSC				Oscillateur quartz moyenne vitesse
;	_HS_OSC				Oscillateur quartz grande vitesse
;	_RC_OSC				Oscillateur  rseau RC
#endif
#ifdef	Fother
	MESSG		"Processor = Fother"
	; add directives
#endif
;-----------------------------------------------------------------------------
;	TEST and DEBUG
;	Comment out the next line [;#define TEST] if no test and debug
;#define	TEST	; TEST and DEBUG mode
;#define	AUTORANGE_TEST	; to display autorange value
;#define	MEASURE1_TEST	; to display autorange value
#ifdef	TEST
	#define	NO_BUSY			; don't test LCD busy flag
	#define	NO_EEPROM_WR	; don't test WR flag
	#define	TEST_MOD		; test MOD parameter
	#define	BIN_TO_DEC		; test BinToDec conversion
	MESSG	"TEST and DEBUG = NO_BUSY+NO_EEPROM_WR"
#endif

;-----------------------------------------------------------------------------
;	RELEASE
#define	V1_4

;-----------------------------------------------------------------------------
;	REGISTERS initial value
;
#define OPTIONVAL	b'00100000'	; Option register value
;b7 RBPU: PORTB Pull-up resistor Enable bit
;	0 = PORTB pull-ups are disabled
;	1 = PORTB pull-ups are enabled (by individual port latch values). 
;b6 INTEDG: Interrupt Edge Select bit
;	1 = Interrupt on rising edge of RB0/INT pin
;	0 = Interrupt on falling edge of RB0/INT pin
;b5 T0CS: TMR0 Clock Source Select bit
;	1 = Transition on RA4/T0CKI pin
;	0 = Internal instruction cycle clock (CLKOUT)
;b4 T0SE: TMR0 Source Edge Select bit
;	1 = Increment on high-to-low transition on RA4/T0CKI pin
;	0 = Increment on low-to-high transition on RA4/T0CKI pin
;b3 PSA: Prescaler Assignment bit
;	1 = Prescaler assigned to the WDT
;	0 = Prescaler assigned to TMR0
;b2,b1,b0: PS2:PS1:PS0: Prescaler Rate Select bits
;Bit Value 	TMR0 Rate 	WDT Rate
;	000 	1 : 2 		1 : 1
;	001 	1 : 4 		1 : 2
;	010 	1 : 8 		1 : 4
;	011 	1 : 16 		1 : 8
;	100 	1 : 32 		1 : 16
;	101 	1 : 64 		1 : 32
;	110 	1 : 128 	1 : 64
;	111 	1 : 256 	1 : 128

#define INTERMASK	b'00000000' ; Interruption mask
; GIE b7=0 : n'autorise pas les interruptions
; EEIE b6=0 : n'autorise pas l'interruption de fin d'criture EEPROM
; T0IE b5=0 : n'autorise pas l'interruption de dbordement du timer0
; INTE b4=0 : n'autorise pas l'interruption de changement de niveau de RB0
; RBIE b3=0 : n'autorise pas l'interruption de changement de niveau de RB'  RB7
; TOIF b2=0 : flag de TOIE non utilis
; INTF b1=0 : flag de INTE valeur initiale
; RBIF b0=0 : flag de RBIE non utilis

#define RESET_V			0x0000		; Address of RESET Vector
#define ISR_V			0x0004		; Address of Interrupt Vector

; PORTB : b7 = Data Bit 7
;		  b6 = Data Bit 6
;		  b5 = Data Bit 5
;		  b4 = Data Bit 4
;		  b3 = Enabled Starts data read/write.
;		  b2 = Read/nWrite Selects read or write. 0: Write, 1: Read
;		  b1 = RS register Selects registers.
;				0: Instruction register (for write) Busy flag:address counter (for read)
;				1: Data register (for write and read)
;		  b0 = inutilis
#define LCD_DATA		PORTB		; LCD data lines interface
#define LCD_DATA_TRIS	TRISB
#define LCD_CTRL		PORTB		; LCD control lines interface

; LCD_CTRL bits
#define LCD_E			3			; LCD Enable control line
#define LCD_RW			2			; LCD Read/Write control line
#define LCD_RS			1			; LCD Register-Select control line

; LCD address line
#define LCD_LINE0		0x000
#define LCD_LINE1		0x040
#define LCD_LINE2		0x014
#define LCD_LINE3		0x054

; LCD digit display
#define DECIMALPOINT	'.'			; decimal symbol : dot, comma, ...

#ifdef XTAL4
	MESSG	"XTAL = 4 MHz"
	#define TIME_BASE 	d'165'	; 500uS delay with 4MHz
	#define DELAY_275u_HIGH 0x01	; 260us delay with 4MHz high counter
	#define DELAY_275u_LOW  0x0B	; 260us delay with 4MHz low counter
	#define DELAY_500m_HIGH 0x64	; 500ms delay with 4MHz high counter
	#define DELAY_500m_LOW  0xC8	; 500ms delay with 4MHz low counter
#endif
#ifdef XTAL10
	MESSG	"XTAL = 10 MHz"
	#define TIME_BASE 	d'249'	; 500uS delay with 10MHz
#endif

; Setting button
#define SetupButton	PORTA,RA0		; Pin 17, 0 = "Setup"
#define NextButton	PORTA,RA1		; Pin 18, 0 = "Next >"
; PROG_FLAGS bit definition
#define Setup_flag		PROG_FLAGS,0x00	; Setup on off
#define SetupIF_flag	PROG_FLAGS,0x01	; Setup IF parameter on off
#define SetupMOD_flag	PROG_FLAGS,0x02	; Setup MOD parameter on off
#define Setupx	PROG_FLAGS,0x03	; Setup x future use
#define Setupy	PROG_FLAGS,0x04	; Setup y future use
#define Count_OK	PROG_FLAGS,0x05	; Count_OK test

; Initial default parameters
#define MODdefault		0		; no IF
#define Number_notLeadingZero 	LCDOption,0x07	; 1 = not leading zero
; NUMBER_FORMAT code
#define END_FORMAT		0xFF
#define	DOT_FORMAT		0x2E
#define	NINE_FORMAT		0x39
#define	ZERO_FORMAT		0x5A

;================================================================================
;								PIC MACRO
;================================================================================

BANK0	macro
		bcf	STATUS,RP0		
		endm

BANK1	macro
		bsf	STATUS,RP0		
		endm
;
Movff32    macro      file1,file2           ; 32 bits file2 = (file1)
           movf       file1,w
           movwf      file2
           movf       file1-1,w
           movwf      file2-1
           movf       file1-2,w
           movwf      file2-2
           movf       file1-3,w
           movwf      file2-3
           endm

;================================================================================
;								LCD MACRO
;================================================================================

IR_WRITE	macro
			bcf	LCD_CTRL, LCD_RS
			bcf	LCD_CTRL, LCD_RW
			endm

BF_READ		macro
			bcf	LCD_CTRL, LCD_RS
			bsf	LCD_CTRL, LCD_RW
			endm

DR_WRITE	macro
			bsf	LCD_CTRL, LCD_RS
			bcf	LCD_CTRL, LCD_RW
			endm

DR_READ		macro
			bsf	LCD_CTRL, LCD_RS
			bsf	LCD_CTRL, LCD_RW
			endm

; Display number with format
LCDDisplay_NumberF macro numberAddress, formatAddress
			movlw	numberAddress		; W = address MSB 
			movwf	FSR					; FSR = address
			movlw	formatAddress  - NUMBER_FORMAT - 1	; start format address in table NumberFormat
			call 	LCDDisplayNumberF	; Display 
			endm

; Display text from TABLE_TEXT
LCDDisplay_Textl macro textAddress
			movlw	textAddress - TABLE_TEXT - 1	; W = Text relative address 
			call 	LCDDisplayText	; Display 
			endm

SET_NOT_LEADING_ZERO	macro
						bsf	Number_notLeadingZero
						endm

CURSOR_POS	macro 					; W register = pos address
			call	LCDSDDA			; Position cursor
			endm

CURSOR_POSl	macro pos
			movlw	pos				; Cursor pos address
			call	LCDSDDA			; Position cursor
			endm

LCDSet_Display_Control macro set
			movlw	set
			call	LCDDmode
			endm

;================================================================================
;							SETUP MACRO
;================================================================================
;__________________________
; Set function Flag
SET_ON		macro
			bsf	Setup_flag
			endm
SET_OFF		macro
			bcf	Setup_flag
			endm
SET_IF_ON	macro
			bsf	SetupIF_flag
			endm
SET_IF_OFF	macro
			bcf	SetupIF_flag
			endm
SET_MOD_ON	macro
			bsf	SetupMOD_flag
			endm
SET_MOD_OFF	macro
			bcf	SetupMOD_flag
			endm

;================================================================================
;							COUNTER MACRO
;================================================================================
;__________________________
; Start stop count
; RA3_START is input 
COUNT_START macro                           
			BANK1
           	bsf	TRISA,RA3
			BANK0
			endm
; RA3_STOP is output 
COUNT_STOP  macro                            
			BANK1
           	bcf	TRISA,RA3
			BANK0
			endm
;================================================================================
;							MACRO END
;================================================================================

;*****************************************************************************
;                   General Purpose registers                        
;*****************************************************************************
	CBLOCK 	RAMStart	; 68 General Purpose registers (SRAM)
	PROG_FLAGS:1		; program flags
	TextIndex:1			; Index to table strings
	TableIndex:1		; Table index
	tempCount:1			; general purpose counter
	;__________________________________________
	; Frequency counter variables
	BFC3:1				; 32 bits frequency MSB
	BFC2:1				; 24 bits frequency
	BFC1:1				; 16 bits frequency 
	BFC0:1				; 08 bits frequency LSB
	countHigh:1
	countLow:1
	TMR0_old:1
	gateHigh:1
	gateLow:1
	PreSC:1		; prescaler
	;_____________________________________
	; IF decimal parameter value 99999 KHz 
	IF4:1  ; High digit MSB
	IF3:1        		
	IF2:1        		
	IF1:1        		
	IF0:1  ; Low digit LSB
	IFindex:1 ; to use in setup
	;_____________________________________
	; IF binary parameter
	IFb3:1		; IF bin MSB
	IFb2:1		; IF bin
	IFb1:1		; IF bin
	IFb0:1		; IF bin LSB
	;_____________________________
	; MOD parameter
	MODindex:1
	;_____________________________________
	; EEPROM variables
	ByteEEcount:1	; EEPROM bytes counter
	;_____________________________________
	; delay.asm Delay subroutine variables
	; 2 bytes reseved
	Delay	; Used in Delayxxx routines
	x_Delay	; Used in x_Delayxxx routines
	;_____________________________________
	; lcd.asm LCD subroutine variables
	; 3 bytes reseved
	LCD_TEMP:1		; LCD subroutines internal use
	LCDOption:1		; internal register don't use
	LCDIndex:1		; general purpose index
	;_____________________________________
	; math.sub Math subroutine variables
	; 24 bytes reseved
	ACa3:1       ; 32-bit Accumulator a, lsb+3, ms-byte
	ACa2:1       ; 32-bit Accumulator a, lsb+2
	ACa1:1       ; 32-bit Accumulator a, lsb+1
	ACa0:1       ; 32-bit Accumulator a, ls-byte

	ACb3:1       ; 32-bit Accumulator b, lsb+3, ms-byte
	ACb2:1       ; 32-bit Accumulator b, lsb+2
	ACb1:1       ; 32-bit Accumulator b, lsb+1
	ACb0:1       ; 32-bit Accumulator b, ls-byte
	    ;
	BCD9:1       ; 10^9, billions
	BCD8:1       ; 10^8
	BCD7:1       ; 10^7
	BCD6:1       ; 10^6, millions
	BCD5:1       ; 10^5
	BCD4:1       ; 10^4
	BCD3:1       ; 10^3, thousands
	BCD2:1       ; 10^2
	BCD1:1       ; 10^1
	BCD0:1       ; 10^0

	bitcnt:	1	; bit count
	digcnt:	1	; digit count
	c_hold:	1	; Carry bit hold
	
	w_temp:	1   	    ; variable used for context saving 
	status_temp:1       ; variable used for context saving
	pclath_temp:1       ; variable used for context saving
	
	ENDC	; End of 68 General Purpose registers (SRAM) max address 0x4F                     

;*****************************************************************************
;                   			EEPROM                        
;*****************************************************************************
	org 	EEPROMStart	
EE_NAME			DE	"F8EOZ   ",0
EE_VERSION		DE	"01.0018122011",0
EE_IF4			DE	0	; MSB
EE_IF3			DE	0	
EE_IF2			DE	0	
EE_IF1			DE	0	
EE_IF0			DE	0	; LSB
EE_MOD			DE	MODdefault
		
;*****************************************************************************
;                      START ON  RESET  
;*****************************************************************************

	org 	RESET_V 	; Adresse de dpart aprs reset
  	goto    start		; Adresse 0: initialiser

;*****************************************************************************
; Text to display
; Relative character address is in W
;*****************************************************************************
TABLE_TEXT
	addwf	PCL ,F ; Jump to character pointed to in W register
MOD_0
    DT   "NO IF"  ; Note the zero termination.
		retlw	0
MOD_1
    DT   "LO+IF"  ; Note the zero termination.
		retlw	0
MOD_2
    DT   "LO-IF"  ; Note the zero termination.
		retlw	0
MOD_3
    DT   "IF-LO"  ; Note the zero termination.
		retlw	0
MODE_LABEL
	DT	"MODE:"
		retlw	0
IF_LABEL
	DT	"IF:"
		retlw	0
UNDERFLOW_ERROR
	DT	"Underflow Error"
		retlw	0
MHz_UNIT
	DT	"MHz"
		retlw	0
TABLE_TEXT_END
		retlw	0
;
	IF ( (TABLE_TEXT & 0x0FF) >= (TABLE_TEXT_END & 0x0FF) )
		MESSG   "==============Warning - User Defined: Table 'TABLE_TEXT' crosses page boundary in computed jump=============="
	ENDIF

MOD_x	; MODindex is in W. Return TABLE_TEXT address into W register
	addwf	PCL ,F ; Jump to character pointed to in W register
	retlw	MOD_0 - TABLE_TEXT - 1
	retlw	MOD_1 - TABLE_TEXT - 1
	retlw	MOD_2 - TABLE_TEXT - 1
	retlw	MOD_3 - TABLE_TEXT - 1

;*****************************************************************************
; Number format
; Relative format code address is in W
;*****************************************************************************
NUMBER_FORMAT
	addwf	PCL ,F ; Jump to character pointed to in W register
FORMAT_MHz
    DT   "Z Z99.999 999"  ; Note the FF termination.
	retlw	0xFF
FORMAT_IF
    DT   "99.999"  ; Note the FF termination.
	retlw	0xFF
FORMAT_MOD
    DT   "9"  ; Note the FF termination.
	retlw	0xFF
NUMBER_FORMAT_END
	retlw	0xFF
;
	IF ( (NUMBER_FORMAT & 0x0FF) >= (NUMBER_FORMAT_END & 0x0FF) )
		MESSG   "==============Warning - User Defined: Table 'NUMBER_FORMAT' crosses page boundary in computed jump=============="
	ENDIF

;================================================================================
;								DELAY SUBROUTINES
;================================================================================
;*****************************************************************************
; Delay_time	= ((TIME_BASE * 3) + 4) * Cycle_time
; TIME_BASE	= (Delay_time - (4 * Cycle_time)) / (3 * Cycle_time)
;
; i.e. (@ 4MHz crystal)
; Delay_time	= ((32 * 3) + 4) * 1us = 100us
; TIME_BASE	= (500uSec - 4*CT) / 3*CT = 165.33	= 165 = TIME_BASE
; CT = (1/clock input)*4 = Instruction Execution Time or Cycle time
; CT = (1/4MHz)*4 = 1 us
; For another clock input, change TIME_BASE
;*****************************************************************************
Delay500	
	movlw	TIME_BASE		; +1		1 cycle
	movwf	Delay			; +2		1 cycle
Delay500_loop	
#ifdef	XTAL10
	; T = 0.1us
	; CT = 4*0.1us = 0.4us
	; TIME_BASE	= (500uSec - 4*0.4) / 5*0.4 = 249.2	= 249 = TIME_BASE
	nop		; wait +1 cycle
	nop		; wait +1 cycle
#endif
	decfsz	Delay, F		; step 1	1 cycle
	goto	Delay500_loop	; step 2	2 cycles
	return					; +4		2 cycles

x_Delay500	
	movwf	x_Delay			; +1		1 cycle
x_Delay500_loop	
	call	Delay500		; step1		wait 500us
	decfsz	x_Delay, F		; step2		1 cycle
	goto	x_Delay500_loop	; step3		2 cycles
	return					; +2		2 cycles
;================================================================================
;								DELAY SUBROUTINES END
;================================================================================

;================================================================================
;								MATH SUBROUTINES
;================================================================================
; Function:   Math routines using PIC16F84
; Created:    29-Jul-2009
; Author:     Brian Beard
; Company:    Lucid Technologies
    ;******************************************************************************
    ;*****       Subroutines.
    ;******************************************************************************
    ;
    ; Quadruple Precision (32-bit) Addition & Subtraction, adapted from AN526.
    ; ACa? and ACb? are each four-byte (32-bit) binary accumulators. 
    ; ACa3 and ACb3 are the MS-Bytes, ACa0 and ACb0 are the LS-Bytes.
    ; Addition (Q_add):    (ACb)+(ACa) --> (ACb), (ACa) is unchanged
    ; Subtraction (Q_sub): (ACb)-(ACa) --> (ACb), (ACa) is negated
    ;
Q_sub   call    neg_ACa       ;First negate ACa, then add
Q_add	movf    ACa0,w
    	addwf   ACb0,f        ;Add lsb
    	btfss   STATUS,C      ;Test for carry
    	goto	q_2           ; go if no carry
    	incf    ACb1,f        ;Add carry to lsb+1
    	btfss	STATUS,Z      ;Test for carry ripple (FF --> 00)
    	goto	q_2           ; and go if no carry
    	incf	ACb2,f        ;Add carry to lsb+2
    	btfsc	STATUS,Z      ;Test for carry ripple
    	incf	ACb3,f        ; add carry to lsb+3, msb
q_2		movf    ACa1,w
    	addwf   ACb1,f        ;Add lsb+1
    	btfss   STATUS,C      ;Test for carry
    	goto	q_4           ; go if no carry
    	incf	ACb2,f        ;Add carry to lsb+2
    	btfsc	STATUS,Z      ;Test for carry ripple
    	incf	ACb3,f        ; add carry to lsb+3, msb
q_4		movf    ACa2,w
    	addwf   ACb2,f        ;Add lsb+2
    	btfsc   STATUS,C      ;Test for carry
    	incf    ACb3,f        ; add carry to lsb+3, msb
    	movf    ACa3,w
    	addwf   ACb3,f        ;Add msb
    	retlw   0
    ;
neg_ACa	comf    ACa0,f       ; complement (ACa)
    	comf    ACa1,f
    	comf    ACa2,f
    	comf    ACa3,f
    	incf    ACa0,f       ; add one
    	btfss   STATUS,Z
    	retlw	0
    	incf    ACa1,f
    	btfss   STATUS,Z
    	retlw	0
    	incf	ACa2,f
    	btfss   STATUS,Z
    	retlw	0
    	incf    ACa3,f
    	retlw   0
    ;
    ;============================================================================
    ; Quadruple Precision (32-bit) unsigned Binary to BCD conversion
    ; BCD9 to BCD0 comprise one ten digit unpacked Binary-Coded-Decimal number.
    ;  The upper nibble of each digit is zero (00008421). BCD9 is the MS-Digit.
    ; ACb3 to ACb0 comprise a four-byte (32-bit) binary accumulator.
    ;  ACb3 is the MS-Byte. 
    ; The 32-bit binary number in ACb in converted to a ten digit BCD number.
    ;
Q_b2bcd
    	clrf	BCD9         ;Clear all bcd digits
    	clrf	BCD8
    	clrf	BCD7
    	clrf	BCD6
    	clrf	BCD5
    	clrf	BCD4
    	clrf	BCD3
    	clrf	BCD2
    	clrf	BCD1
    	clrf	BCD0
    	movlw	D'32'        ;Outer loop
    	movwf	bitcnt       ; bit counter
b2bcd1
    	rlf	ACb0,f       ;Shift 32-bit accumulator
    	rlf	ACb1,f       ; left to 
    	rlf	ACb2,f       ;  put ms-bit 
    	rlf	ACb3,f       ;   into Carry
    	movlw	BCD0         ;Point to address of least 
    	movwf	FSR          ; significant bcd digit
    	movlw	D'10'        ;Inner loop
    	movwf	digcnt       ; digit counter
b2bcd2
    	rlf	INDF,f       ;Shift Carry into bcd digit
    	movlw	D'10'        ;Subtract ten from digit then
    	subwf	INDF,w       ; check and adjust for decimal overflow
    	btfsc   STATUS,C     ;If Carry = 1 (result >= 0)
    	movwf	INDF         ; adjust for decimal overflow
    	decf	FSR,f        ;Point to next bcd digit
    	decfsz	digcnt,f     ;Decrement digit counter
    	goto	b2bcd2       ; - go if digcnt > 0
    	decfsz	bitcnt,f     ;Decrement bit counter
    	goto	b2bcd1       ; - go if bitcnt > 0
    	retlw   0
    ;
    ;============================================================================
    ; Quadruple Precision (32-bit) unsigned BCD to Binary conversion.
    ; BCD9 to BCD0 comprise one ten digit unpacked Binary-Coded-Decimal number.
    ;  The upper nibble of each digit is zero (00008421). BCD9 is the MS-Digit.
    ; ACb3 to ACb0 comprise a four-byte (32-bit) binary accumulator.
    ;  ACb3 is the MS-Byte. 
    ; The 10-digit BCD number is converted to a 32-bit binary number in ACb.
    ;
Q_bcd2b
    	movlw	D'32'         ;Outer loop
    	movwf	bitcnt        ; bit counter
bcd2b1
    	movlw	BCD9          ;Set pointer for
    	movwf	FSR           ; most significant digit
    	movlw	D'10'         ;Inner loop
    	movwf	digcnt        ; digit counter
    	movlw	0x05          ;Weight of next MS-Digit's carry bit
    	clrf	c_hold        ;0 --> carry_hold
bcd2b2
    	bcf	STATUS,0      ;Clear Carry bit
    	rrf	INDF,f        ;Right shift 0 into bcd digit
    	rlf	c_hold,f      ;Carry --> c_hold,0
    	btfsc	c_hold,1      ;Add five if carry from previous rotate is
    	addwf	INDF,f        ; set, else no change
    	incf	FSR,f         ;Advance digit pointer
    	decfsz	digcnt,f      ;Decrement digit counter,
    	goto	bcd2b2        ; repeat if > 0
    	rrf	c_hold,f      ;Last carry from digit rotate --> C 
    	rrf	ACb3,f        ;Ripple Carry bits
    	rrf	ACb2,f        ; from MSB to
    	rrf	ACb1,f        ;  LSB of 
    	rrf	ACb0,f        ;   32-bit binary accumulator
    	decfsz	bitcnt,f      ;Decrement bit counter,
    	goto	bcd2b1        ; repeat if > 0
    	retlw   0
;================================================================================
;								MATH SUBROUTINES END
;================================================================================

;================================================================================
;								LCD SUBROUTINES 
;================================================================================
;-----------------------------------------------------------------
; The HD44780U has two 8-bit registers, an instruction register (IR) and a data register (DR).
; The IR stores instruction codes, such as display clear and cursor shift, and address information for display
; data RAM (DDRAM) and character generator RAM (CGRAM). The IR can only be written from the MPU.
; The DR temporarily stores data to be written into DDRAM or CGRAM and temporarily stores data to be
; read from DDRAM or CGRAM. Data written into the DR from the MPU is automatically written into
; DDRAM or CGRAM by an internal operation. The DR is also used for data storage when reading data
; from DDRAM or CGRAM. When address information is written into the IR, data is read and then stored
; into the DR from DDRAM or CGRAM by an internal operation. Data transfer between the MPU is then
; completed when the MPU reads the DR. After the read, data in DDRAM or CGRAM at the next address is
; sent to the DR for the next read from the MPU. By the register selector (RS) signal, these two registers can
; be selected (Table below).
;-----------------------------------------------------------------
; Table 1 Register Selection
; RS R/W Operation
; 0  0   IR write as an internal operation (display clear, etc.)
; 0  1   Read busy flag (DB7) and address counter (DB0 to DB6)
; 1  0   DR write as an internal operation (DR to DDRAM or CGRAM)
; 1  1   DR read as an internal operation (DDRAM or CGRAM to DR)
;-----------------------------------------------------------------
;*****************************************************************************
; LCDinit
; Initilize LC-Display Module
; Should be modified to your needs (i.e. display type, cursor on/off, etc.)
;*****************************************************************************
LCDinit
							; Busy-flag is not yet valid
							;---------------------------
	clrf	LCD_CTRL		; ALL PORT output should output Low.
	; Power-up delay
	; Datasheet LCD JHD162A.pdf: wait time > 4.5 ms after VDD > 4.5 V
	movlw	0x1E
	call	x_Delay500		; 30 * 0.5mS = 15mS
							; Busy Flag should be valid AFTER the following
							; else, increase delay above
							;----------------------------------------------
	; Function set
	; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
	; Bit value    0  0   0   0   1   0
	; Sets to 4-bit operation. In this case, operation is
	; handled as 8 bits by initialization, and only this instruction
	; completes with one write.
	movlw	B'00100000'		; FUNCTION SET 4 bits
	movwf	LCD_DATA 
	IR_WRITE
	bsf		LCD_CTRL, LCD_E	; LCD E-line High
	bcf		LCD_CTRL, LCD_E	; LCD E-line Low , crire sur front descendant
	
	;--------------------------------------------------------
	;..........EVERY INSTRUCTION WILL BE OF 2 CYCLE FROM HERE
	;--------------------------------------------------------
	; Function set
	; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
	; Bit value    0  0   0   0   1   0   N   F   *   *
	; Sets to 8-bit operation and selects 2-line display and 5 * 8 dot character font.
	; The number of display lines and character font cannot be changed after this point.
	; N = 1: 2 lines, N = 0: 1 line
	; F = 1: 5x10 dots, F = 0: 5x8 dots	
	movlw	B'00101000'		; 2 lines, 5x8 dots
	call	LCDputCmd4
	
	; Display on/off control 0.040 ms
	; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
	; Bit value    0  0   0   0   0   0   1   D   C   B 
	; Sets entire display (D) on/off,
	; cursor on/off (C), and
	; blinking of cursor position character (B).
	movlw	B'00000000'		; disp.off, curs.off, no-blink
	call 	LCDDmode
	
	; Clear display 1.64 ms
	; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
	; Bit value    0  0   0   0   0   0   0   0   0   1
	; Clears entire display and sets DDRAM address 0 in address counter.
	call	LCDclear
	
	movlw	B'00000100'		; disp.on, curs.off, blink off
	call	LCDDmode
	
	; Entry mode set 0.040 ms
	; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
	; Bit value    0  0   0   0   0   0   0   1   I/D S
	; Sets cursor move direction and specifies display shift.
	; These operations are performed during data write and read.
	; I/D = 1: Increment
	; I/D = 0: Decrement
	; S = 1: Accompanies display shift
	; I/D: Increments (I/D = 1) or decrements (I/D = 0) the DDRAM address by 1 when a character code is
	; written into or read from DDRAM.
	; The cursor or blinking moves to the right when incremented by 1 and to the left when decremented by 1.
	; The same applies to writing and reading of CGRAM.
	; S: Shifts the entire display either to the right (I/D = 0) or to the left (I/D = 1) when S is 1. The display does
	; not shift if S is 0.
	; If S is 1, it will seem as if the cursor does not move but the display does. The display does not shift when
	; reading from DDRAM. Also, writing into or reading out from CGRAM does not shift the display.
	movlw	B'00000010'		; auto-inc (shift-cursor)
	call	LCDEmode
	
	return

;*****************************************************************************
; LCDputCmd4
; Sends command to LCD 4 bits
; Required command must be in W
;*****************************************************************************
LCDputCmd4		
	movwf	LCD_TEMP		; Command to be sent is in W
	call	LCDbusy			; Wait for LCD to be ready

	movf	LCD_TEMP,W		; restore w
	call 	LCDputCMDnibble	; 1st nibble: high order bits

	swapf	LCD_TEMP,W		; 2nd nibble : SWAP NIBBLES AND STORE IN W REGISTER
	call 	LCDputCMDnibble	; 2nd nibble: : low order bits

	return

;*****************************************************************************
; LCDputCMDnibble
; Sends one nibble
; Required nibble must be in W
;*****************************************************************************
LCDputCMDnibble
	andlw	B'11110000' 	; strip lower bits, garde DB7, DB6, DB5, DB4
	movwf	LCD_DATA		; Send data to LCD
	IR_WRITE
	bsf		LCD_CTRL, LCD_E	; LCD E-line High
	bcf		LCD_CTRL, LCD_E	; LCD E-line Low: 1st cycle complete

	return

;********************************************************************************
; LCDbusy																	
; Returns when LCD busy-flag is inactive									
;********************************************************************************
; Busy Flag (BF)
; When the busy flag is 1, the HD44780U controller is in the internal
; operation mode, and the next instruction will not be accepted.
; When RS = 0 and R/W = 1 (Table 1 datasheet HD44780), the busy flag
; is output to DB7. The next instruction must be written after
; ensuring that the busy flag is 0.
; When an instruction is being executed for internal operation, 
; no instruction other than the busy flag/address read instruction
; can be executed. Because the busy flag is set to 1 while an
; instruction is being executed, check it to make sure it is 0 before
; sending another instruction from the MPU.
; Note that two cycles are needed for the busy flag check as well as 
; for the data transfer. The 4-bit operation is selected by the program.
LCDbusy
#ifdef	NO_BUSY
	return	; no busy test
#endif

	BANK1
	movlw	B'11110000'		; select LCD_DATA for input set RB7 to RB4 =1
							; select LCD_CTRL for output set RB3 to RB1 =0 (unchanged)
	movwf	TRISB			; SET RB7 to RB4 INPUT RB3 to RB0 OUTPUT
	BANK0
	BF_READ
	bsf		LCD_CTRL, LCD_E	; LCD E-line High
	movf	LCD_DATA, W		; Read busy flag + DDram address
	bcf		LCD_CTRL, LCD_E	; LCD E-line Low : 1fst cycle complete

	; 2nd cycle 4-bit data 
	BF_READ
	bsf		LCD_CTRL, LCD_E	; LCD E-line High
	bcf		LCD_CTRL, LCD_E	; LCD E-line Low : 2nd cycle complete
	
	; Busy flag is in DB7
	andlw	0x80			; Check Busy flag, High = Busy
	btfss	STATUS, Z
	goto	LCDbusy			; LOOP IF BUSY : Tant que DB7 = BF = 1
LCDnotBusy	
	bcf		LCD_CTRL, LCD_RW
	BANK1					;NOT BUSY SO MAKE PORTB Output
	movlw	0x0000
	movwf	TRISB			; Set PORTB for output
	BANK0
		
	return

;*****************************************************************************
; LCDDmode
; Sets display control.
; Display on/off control 0.040 ms
; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
; Bit value    0  0   0   0   0   0   1   D   C   B 
; Sets entire display (D) on/off,
; cursor on/off (C), 
; blinking of cursor position character (B).
; Required display mode must be set in W
;  b0	: 0 = cursor blink off	1 = cursor blink on
;  b1	: 0 = cursor off	1 = cursor on
;  b2	: 0 = display off	1 = display on (display data remains in DDRAM)
;  b3-7	: don't care
;*****************************************************************************
LCDDmode
	andlw	B'00000111'	; Strip upper bits
	iorlw	B'00001000'	; Function set
	call	LCDputCmd4
	
	return

;*****************************************************************************
; LCDclear
; Clears display and returns cursor to home position (upper-left corner).
; Clear display
; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
; Bit value    0  0   0   0   0   0   0   0   0   1 
; Clears entire display and sets DDRAM address 0 in address counter.;
;*****************************************************************************
LCDclear
	movlw	0x01
	call	LCDputCmd4
	
	return

;*****************************************************************************
; LCDEmode
; Sets entry mode of display.
; Entry mode set (37 us)
; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
; Bit value    0  0   0   0   0   0   0   1   I/D S 
; Sets cursor move direction and specifies display shift.
; These operations are performed during data write and read.
; I/D = 1: Increment
; I/D = 0: Decrement
; S = 1: Accompanies display shift
; I/D: Increments (I/D = 1) or decrements (I/D = 0) the DDRAM address by 1 when a character code is
; written into or read from DDRAM.
; The cursor or blinking moves to the right when incremented by 1 and to the left when decremented by 1.
; The same applies to writing and reading of CGRAM.
; S: Shifts the entire display either to the right (I/D = 0) or to the left (I/D = 1) when S is 1. The display does
; not shift if S is 0.
; If S is 1, it will seem as if the cursor does not move but the display does. The display does not shift when
; reading from DDRAM. Also, writing into or reading out from CGRAM does not shift the display.
; Required entry mode must be set in W
;  b0	: 0 = no display shift	1 = display shift
;  b1	: 0 = auto-decrement	1 = auto-increment
;  b2-7	: don't care
;*****************************************************************************
LCDEmode
	andlw	B'00000011'		; Strip upper bits
	iorlw	B'00000100'		; Function set
	call	LCDputCmd4

	return

;*****************************************************************************
; LCDSDDA
; Sets the Display-Data-RAM address. DDRAM data is read/written after
;  this setting.
; Set DDRAM address (37 us)
; Instruction  RS R/W DB7 DB6 DB5 DB4 DB3 DB2 DB1 DB0
; Bit value    0  0   1   ADD ADD ADD ADD ADD ADD ADD
; Sets DDRAM address. DDRAM data is sent and received after this setting.
; Required DDRAM address must be set in W
;  b0-6	: required DDRAM address
;  b7	: don't care
;*****************************************************************************
LCDSDDA
	iorlw	B'10000000'		; Function set
	call	LCDputCmd4

	return

;*****************************************************************************
; LCDputChar4
; Sends character to LCD
; Required character must be in W
;*****************************************************************************
LCDputChar4
	movwf	LCD_TEMP		; Command to be sent is in W
	call	LCDbusy			; Wait for LCD to be ready

	movf	LCD_TEMP,W		; restore w
	call 	LCDputCharNibble; 1st nibble: high order bits

	swapf	LCD_TEMP,W		; 2nd nibble : SWAP NIBBLES AND STORE IN W REGISTER
	call 	LCDputCharNibble; 2nd nibble: low order bits
	
	return

;*****************************************************************************
; LCDputCharNibble
; Sends one nibble to LCD
; Required nibble must be in W
;*****************************************************************************
LCDputCharNibble
	andlw	B'11110000' 	; strip lower bits, garde DB7, DB6, DB5, DB4
	movwf	LCD_DATA		; Send data to LCD
	DR_WRITE
	bsf		LCD_CTRL, LCD_E	; LCD E-line High
	bcf		LCD_CTRL, LCD_E	; LCD E-line Low: 1st cycle complete
	
	return

;********************************************************************************
; DisplayString
; Display a string								
; Required string address must be in FSR
; end of string = 0
;********************************************************************************
DisplayString
    movf    INDF,W				; W = character n to display
	btfsc	STATUS,Z			; string is empty ?
	return						; yes, no character displayed

DisplayStringLoop
	call	LCDputChar4			; Display character
	incf	FSR,f				; next address character
	movf	INDF,W				; W = next character
	btfss	STATUS,Z			; end of string ?
	goto 	DisplayStringLoop	; no
	
	return

;*****************************************************************************
; Display a number at cursor position using a table NUMBER_FORMAT
; Format sample: ZZZ Z9 or 99 or 9.999 or 9 999 999 ....
; Start address character must be in FSR
; Start address table NUMBER_FORMAT must be in W
; Number_notLeadingZero = 0 => Leading zeros are not displayed
; Number_notLeadingZero = 1 => zero is not Leading zero
;______________________________________________________
; ASCII value	x30 x31 x32 x33 x34 x35 x36 x37 x38 x39 
; number		0	1	2	3	4	5	6	7	8	9
;______________________________________________________
;*****************************************************************************
LCDDisplayNumberF
	movwf	LCDIndex				; Holds format address in table NUMBER_FORMAT
	clrf	LCDOption
LCDDisplayNumberFLoop
	; is leading Zero ?
	movfw	LCDIndex
	call	NUMBER_FORMAT
	xorlw	0x5A					; Check if "Z" format
	btfsc	STATUS, Z				 
	goto	Z_format				; yes, ====>
	; is nine ?
	movfw	LCDIndex
	call	NUMBER_FORMAT
	xorlw	0x39					; Check if "9" format
	btfsc	STATUS, Z				
	goto	nine_format				; yes, ====>
	; is dot ?
	movfw	LCDIndex
	call	NUMBER_FORMAT
	xorlw	0x2E					; Check if "." format
	btfsc	STATUS, Z				
	goto	dot_format				; yes, ====>
	; is end ?
	movfw	LCDIndex
	call	NUMBER_FORMAT
	xorlw	0xFF					; Check if at end of format
	btfsc	STATUS, Z				
	goto    LCDDisplayNumberFEnd	; yes, end =====================>            
	
space_format	; default
	btfss	Number_notLeadingZero	; is a number 1 to 9 already displayed ?
	goto	nextIndexFormat			; no, no space to insert
	movlw	' '						; yes, insert a space
	call	LCDputChar4				; Display character
	goto	nextIndexFormat
dot_format
	SET_NOT_LEADING_ZERO			; to avoid future Z format
	movlw	DECIMALPOINT			; insert decimal point
	call	LCDputChar4				; Display character
	goto	nextIndexFormat
nine_format
	SET_NOT_LEADING_ZERO			; to avoid future Z format
	goto	displayASCII
Z_format
	btfsc	Number_notLeadingZero	; is a number 1 to 9 already displayed ?
	goto	displayASCII			; yes, display digit

	movf	INDF, f					; INDF -> INDF, set STATUS bit Z
	btfsc	STATUS, Z				
	goto	nextDigit				; yes digit = 0, leading zero not displayed
	SET_NOT_LEADING_ZERO			; no, display digit and avoid future Z format
displayASCII
    movf    INDF,W     				; Digit -> W
    iorlw   030h					; ASCII value mask
	call	LCDputChar4				; Display character
nextDigit
	incf	FSR, f
nextIndexFormat
	incf	LCDIndex,f				; Point to next character
	goto	LCDDisplayNumberFLoop
LCDDisplayNumberFEnd

	return

;*****************************************************************************
; Display a text at cursor position using a table 
; Required start address character must be in W
;*****************************************************************************
LCDDisplayText
	movwf	TableIndex			; Holds text address
LCDDisplayTextLoop
	movfw	TableIndex		
	call	TABLE_TEXT
	andlw	0xFF				; Check if at end of text
	btfsc	STATUS, Z			; (zero returned at end)
	goto    LCDDisplayTextEnd	; end =====================>            
	call	LCDputChar4			; Display character
	incf	TableIndex,f		; Point to next character
	goto	LCDDisplayTextLoop
LCDDisplayTextEnd

	return

;================================================================================
;								LCD SUBROUTINES END
;================================================================================

;================================================================================
;								OPTIONS MANAGEMENT
;================================================================================

;********************************************************************************
; Load options from EEPROM									
;********************************************************************************
LoadOptions
	movlw	IF0				; IF destination address
	movwf	FSR				; into FSR
	movlw	5				; read 5 bytes
	movwf	ByteEEcount	
	movlw	EE_IF0+1		; relative start address
	movwf	EEADR			; into EEADR
ReadEE_IFloop    
	decf	EEADR,f			; input address
	call	ReadEE			; read one byte from EEPROM
	movf	EEDATA,W		; data ouput
	movwf	INDF			; into IF destination address
	decf	FSR,f			; next FI byte address
	decfsz	ByteEEcount,f
	goto	ReadEE_IFloop

	call	IFtoIFb			; IF binary conversion

	movlw	EE_MOD			; MOD input address
	movwf	EEADR
	call	ReadEE
	movf	EEDATA,W		; data ouput
	movwf	MODindex		; into MOD parameter

	return

;********************************************************************************
; IF decimal KHz to binary convertion Hz								
;********************************************************************************
IFtoIFb
	clrf	BCD9
	clrf	BCD8
	movfw	IF4
	movwf	BCD7
	movfw	IF3
	movwf	BCD6
	movfw	IF2
	movwf	BCD5
	movfw	IF1
	movwf	BCD4
	movfw	IF0
	movwf	BCD3
	clrf	BCD2
	clrf	BCD1
	clrf	BCD0
	call    Q_bcd2b			; IF binary conversion
	Movff32	ACb0, IFb0	; store 32 bits IF
	return

;********************************************************************************
; LoadEEstring
; Load string from EEPROM								
; Required string address destination must be in FSR
; Required EEPROM start address destination must be in W
; end of string = 0
;********************************************************************************
LoadEEstring
	movwf	EEADR			; into EEADR
LoadEEstringLoop
	call	ReadEE			; read one byte from EEPROM
	movf	EEDATA,W		; data output
	btfsc	STATUS, Z		; zero returned at end
	goto	LoadEEstringEnd

	movwf	INDF			; data into destination address
	incf	FSR,f			; next destination byte address
	incf	EEADR,f			; next input address
	goto	LoadEEstringLoop
LoadEEstringEnd
	movwf	INDF			; zero into destination address
	return

;********************************************************************************
; SaveOptions to EEPROM									
;********************************************************************************
SaveOptions
	movlw	IF0				; IF input address
	movwf	FSR				; into FSR
	movlw	5				; write 5 bytes
	movwf	ByteEEcount	
	movlw	EE_IF0+1		; relative output address
	movwf	EEADR			; into EEADR
WriteEE_IFloop    
	decf	EEADR,f			; output address
	movf	INDF,W			; IF into EEDATA
	movwf	EEDATA			
	call	WriteEE			; read one byte from EEPROM
	decf	FSR,f			; next FI byte address
	decfsz	ByteEEcount,f
	goto	WriteEE_IFloop

	movf	MODindex,W		; MOD into EEDATA
	movwf	EEDATA			
	movlw	EE_MOD			; MOD output address
	movwf	EEADR
	call	WriteEE			; read one byte from EEPROM

	return

;*****************************************************************************
; ReadEE
; Read n character from EEPROM address
; Required relative start EEPROM address must be in EEADR
; Required number of data must be in ByteEEcount
; Required destination address must be in FSR
;*****************************************************************************
ReadEE    
	BANK1
	bsf		EECON1,RD	; read
	BANK0
	return

;*****************************************************************************
; WriteEE
; This routine writes to the eeprom. It is assumed that the EEADR and EEDATA
; registers are loaded with the correct values before calling this routine
;*****************************************************************************
WriteEE
	BANK1
	bcf EECON1,EEIF 	;disable interrupt write to EEPROM
	bsf EECON1,WREN 	;enable write to EEPROM
	movlw 55h
	movwf EECON2
	movlw 0xaa
	movwf EECON2
	bsf EECON1,WR		; set WR = write EEPROM
#ifndef	TEST
WriteEEwait
	btfsc EECON1,WR		; loop <===============
	goto $-1			; loop until WR = 0 ===
#endif
	bcf EECON1,WREN 	; disable write to EEPROM
	BANK0
	return 

;================================================================================
;								OPTIONS MANAGEMENT END
;================================================================================

;================================================================================
; 										SETUP
;================================================================================
; Is Setup button pushed 
;	IF parameter = 99.999 MHz
;	MOD parameter = 0 : NO IF
;	MOD parameter = 1 : LO + IF
;	MOD parameter = 2 : LO - IF
;	MOD parameter = 3 : IF - LO
;
;	For Setup use push button SET and Next (>)
;	1st time, for Setup, push SET button:
;	On 1st line "Setup" is displayed
;	On 2nd line IF value is displayed with flashing cursor on first digit
;	For next value push Next button and so on 
;	For next digit press Set button
;	MOD parameter starts by pushing Set button on last FI digit
;	On 2nd line MOD value is displayed with flashing cursor on MOD parameter
;	For next value push Next button and so on 
;	To save parameters into EEPROM push Set button
;********************************************************************************
isSetup
LoopSetup	
	btfsc	SetupButton		; Button set is pushed ?
	goto	isNextSetup		; no
	movlw	0xC8			; x delay factor = 200d
	call	x_Delay500		; 200 * 0.5mS = 100mS
LoopSetButton
	btfss	SetupButton		; Button set is pushed ?
	goto	LoopSetButton
	movlw	0xC8			; x delay factor = 200d
	call	x_Delay500		; 200 * 0.5mS = 100mS

	btfsc	Setup_flag		; Setup started ?
	goto	SetIsOn			; yes
	call	StartSetup		; no : start
	goto	isNextSetup
SetIsOn
	call 	SetupIsOn
isNextSetup
	btfss	Setup_flag			; End setup ?
	goto	isSetupExit			; yes
NextButtonPushed
	btfsc	NextButton			; Button next is pushed ?
	goto	LoopSetup			; no : wait
	movlw	0xC8				; x delay factor = 200d
	call	x_Delay500			; 200 * 0.5mS = 100mS
LoopNextButton
	btfss	NextButton			; Button next is on
	goto	LoopNextButton
	movlw	0xC8				; x delay factor = 200d
	call	x_Delay500			; 200 * 0.5mS = 100mS

	btfss	SetupIF_flag		; IF option running ?
	goto	NextMOD				; no : see next option
NextIF
	call	IncIFdigit			; IF option inc digit
	goto	LoopSetup		
NextMOD
	btfss	SetupMOD_flag		; MOD option is running ?
	goto	NextFlag			; no
	call	IncMODdigit			; MOD option inc digit
	goto	LoopSetup	
NextFlag
	goto	LoopSetup	
isSetupExit
	return

;********************************************************************************
; StartSetup	SETUP ON ==> SETUP BUTTON									
;********************************************************************************
StartSetup
	SET_ON

	; Clears entire display and sets DDRAM address 0 in address counter.
	call	LCDclear
	
	LCDSet_Display_Control  B'00000111'		; disp.on, curs.on, blink on

	SET_IF_ON

	call 	DisplayIFparameter

	movlw	0x03			; store 1st cursor position
	movwf	tempCount
	call 	IFcursorPos
	
	movlw	IF4				; store IF MSB address
	movwf	IFindex
	
	movlw	FORMAT_IF - NUMBER_FORMAT - 1	; store relative address IF format
	movwf	TableIndex

	return

;********************************************************************************
; SetupIsOn
;********************************************************************************
SetupIsOn
	btfss	SetupIF_flag	; SET IF is ON ?
	goto	SetupIsOn_Next	; no, next parameter
IFset
	call	IFsetIsOn
	return
SetupIsOn_Next
	SET_MOD_OFF
	SET_OFF

	call SaveOptions		; save parameter into EEPROM

	call	IFtoIFb			; IF binary conversion
	
	call	LCDclear ; Clears entire display and sets DDRAM address 0 in address counter.

	LCDSet_Display_Control  B'00000100'		; disp.on, curs.off, blink off

	return

;********************************************************************************
; IFsetIsOn
; Flashing cursor on next digit									
; If last digit skip to MOD parameter
;********************************************************************************
IFsetIsOn

loop_format
	incf	TableIndex, f		; next address format
	; is end ?
	movfw	TableIndex
	call	NUMBER_FORMAT
	xorlw	END_FORMAT			; Check if at end of format
	btfsc	STATUS, Z				
	goto    IFsetOK				; yes, end =====================>            
	; is nine ?
	movfw	TableIndex
	call	NUMBER_FORMAT
	xorlw	NINE_FORMAT			; Check if "9" format
	btfsc	STATUS, Z				
	goto	next_FIdigit			; yes, ====>
	; default other format
	incf	tempCount, f		; cursor on next digit
	goto	loop_format
next_FIdigit
	incf	IFindex, f			; next FI byte address
	incf	tempCount, f		; cursor on next digit
	call 	IFcursorPos
	return
IFsetOK 
	SET_IF_OFF
	SET_MOD_ON

	; Clears entire display and sets DDRAM address 0 in address counter.
	call	LCDclear

	LCDSet_Display_Control  B'00000111'		; disp.on, curs.on, blink on

	call	DisplayMODparameter
	return

;********************************************************************************
; DisplayIFparameter
; display IF parameter on line 1									
;********************************************************************************
DisplayIFparameter
	CURSOR_POSl LCD_LINE1	; Position cursor leftmost on current line
	
	LCDDisplay_Textl	IF_LABEL	; display IF text

	LCDDisplay_NumberF IF4, FORMAT_IF
	
	movlw	' '	
	call	LCDputChar4			; Display character

	LCDDisplay_Textl	MHz_UNIT	; display unit text

	return

;********************************************************************************
; IncIFDigit
; Increment digit counter									
; Required address digit counter must be in FSR
; Counter = 0, 1, 2, ..., 9, 0, 1, 2 ... and so on
;********************************************************************************
IncIFdigit
	movfw	IFindex		
	movwf	FSR					; FSR = IFindex address	
	movlw	0x0A				; maximum value
	call	IncDigitCounter
    movf    INDF,W     			; Digit -> W
    iorlw   030h				; ASCII value mask
	call	LCDputChar4			; Display character
	call 	IFcursorPos
	
	return

;********************************************************************************
; IFcursorPos
; Cursor on IF digit									
; Required IF index must be in IFindex
;********************************************************************************
IFcursorPos
	movfw	tempCount
	addlw	LCD_LINE1		; coursor pos = address line + pos cursor
	CURSOR_POS				; Position cursor at W
	return

;********************************************************************************
; DisplayMODparameter
; display MOD parameter on line 1									
;********************************************************************************
DisplayMODparameter
	CURSOR_POSl LCD_LINE1	; Position cursor leftmost on current line
	
	LCDDisplay_Textl	MODE_LABEL	; display MODE text

	LCDDisplay_NumberF MODindex, FORMAT_MOD
		
	movlw	' '	
	call	LCDputChar4		; Display one space

	movf	MODindex, w		; get address from TABLE_TEXT
	call	MOD_x			; return address MODE option text from TABLE_TEXT in W
	call 	LCDDisplayText	; display text

	CURSOR_POSl LCD_LINE1+0x05	; Cursor on MOD digit restore address line + pos cursor
	
	return

;********************************************************************************
; IncMODDigit
; Increment digit counter									
; Required address digit counter must be in FSR
; Counter = 0, 1, 2, ..., 9, 0, 1, 2 ... and so on
;********************************************************************************
IncMODdigit
	movlw	MODindex			
	movwf	FSR					; FSR = MODindex address		
	movlw	0x04				; maximum value
	call	IncDigitCounter
	call 	DisplayMODparameter
	return

;********************************************************************************
; IncDigitCounter
; Increment digit counter									
; Required address digit counter must be in FSR
; Required maximum value must be in W
; If W = 10d counter = 0, 1, 2, ..., 9, 0, 1, 2 ... and so on
; If W = 04d counter = 0, 1, 2, 3, 0, 1, 2, 3, 0 ... and so on
;********************************************************************************
IncDigitCounter
	incf	INDF,f		; Digit += 1
	xorwf	INDF,W		; If digit = maximum value then W = 0
	btfsc	STATUS, Z	; skip if digit not = maximum value
	clrf	INDF		; Digit = 0
	
	return

;================================================================================
; 									SETUP END
;================================================================================

;================================================================================
;								FREQUENCY COUNTER
;================================================================================
;********************************************************************************
; 1) Autorange with delay = 275 us and prescaler = 256
; 2) Set prescaler
; 3) Measure BFC with delay = 0.5 ms and prescaler = n
; 4) Compute BFC = BFC x 2 x 2^(n+1)
;********************************************************************************
CountFrequency
	;______________________________________________________________
	; 1) Autorange count parameters delay = 275 us, prescaler = 256
	movlw	DELAY_275u_HIGH		;0.000275 s gate
	movwf	gateHigh
	movlw 	DELAY_275u_LOW
	movwf	gateLow
	movlw	B'00100111'	; for PRESC = 256, TOCS=1, TOSE=0, PSA=0
	call	Measure
#ifdef	AUTORANGE_TEST
	return	; to display autorange value
#endif

	;______________________________________________________________
	; 2) Set prescaler
	; BFC0  = frequency in MHz
	; Frequency = 1 MHz => BFC0 = 275/256 = 1
	; Frequency = 2 MHz => BFC0 = 275/256 = 2
	; ...
	; >64 MHz 64:1 - 01xx xxxx - PS=5
	; >32 MHz 32:1 - 001x xxxx - PS=4
	; >16 MHz 16:1 - 0001 xxxx - PS=3
	; > 8 MHz  8:1 - 0000 1xxx - PS=2
	; > 4 MHz  4:1 - 0000 01xx - PS=1
	; > 2 MHz  2:1 - 0000 001x - PS=0

	bsf		BFC0, 0x01	; for frequency < 2 MHz
	movlw	0x08			
	movwf	PreSC
PreSC_loop
	decf	PreSC, f  
	rlf		BFC0,f
	btfss	STATUS, C
	goto	PreSC_loop

	;______________________________________________________________
	; 3) Measure BFC parameters : delay = 0.5 ms,d prescaler = n
	movlw	DELAY_500m_HIGH		;0.1 s gate
	movwf	gateHigh
	movlw 	DELAY_500m_LOW
	movwf	gateLow
	movlw	B'00100000' ; TOCS=1, TOSE=0, PSA=0
	addwf	PreSC,w 	; PRESC = n
	call	Measure
#ifdef	MEASURE1_TEST
	return	; to display measure value
#endif

	;______________________________________________________________
	; 4) Compute BFC = BFC x 2 x 2^(n+1)
	call	x_2		; 0.500 s x 2 = 1s
	; BFC = BFC x 2^(n+1)
	movfw	PreSC
	movwf	tempCount		
	incf	tempCount,f	
PreSC_offset_loop
	call	x_2				; BFC = BFC * 2
	decfsz	tempCount,f
	goto	PreSC_offset_loop
	return

;********************************************************************************
; BFC 32 Bit x 2: rotate left BFC0 to BFC-3
;********************************************************************************
x_2
	bcf	STATUS,C
	rlf	BFC0,f
	rlf	BFC0-1,f
	rlf	BFC0-2,f
	rlf	BFC0-3,f

	return

;********************************************************************************
; Parameters : gate time high = p, gate time low = n
; Gate time = 4p + (10+4)np + (3+8)p(n-1) + 2p + (3+2)(p-1) + 2 + 3
; One step = 25 us with XTAL = 4 MHz
; 275 us = 1 x 12 x 25 us => p = 1, n = 12
; 500 ms = 100 x 200 x 0.025 ms => p = 100, n = 200
;********************************************************************************
Measure
	BANK1
	movwf	OPTION_REG 	; 
	BANK0
	clrf	BFC0-1 		; clear 16 bits counter 
	clrf	BFC0-2 		; clear 24 bits counter 
	clrf	BFC0-3 		; clear 32 bits counter 

	bcf		PORTA,0x03
	movf	gateHigh,w	
	movwf	countHigh
	clrf	TMR0_old
	clrf	TMR0
	COUNT_START	; start count
M05	movfw	gateLow		; |
	movwf	countLow	; | = 4p
	goto	M20			; |
M10
	goto M05			; = 2(p-1)

M15	nop					; |
	nop					; |
	nop					; |
	nop					; |  = 8p(n-1)
	nop					; |
	nop					; |
	nop					; |
	nop					; |	
	
M20	movf	TMR0,w		; TMR0 rollover ?	; |
	subwf	TMR0_old,f	; TMR0_old - TMR0	; |
	btfss	STATUS,Z	; 0 = no change		; |
	goto	M25								; | 
	nop										; |
	nop										; |
	nop										; |
	goto	M30								; |
M25	btfsc	STATUS,C	; TMR0 < TMR0_old ?	; | = (10+4)np
	incf	BFC0-1,f	; 16 bits counter	; |
	btfsc	STATUS,Z						; |
	incf	BFC0-2,f	; 24 bits counter	; |
M30	movwf	TMR0_old						; |
	nop										; |
	nop										; |
	nop										; |
	nop										; |
	decfsz	countLow,f		; = 2p
	goto	M15				; = 3p(n-1)
	decfsz	countHigh,f		; = 2
	goto	M10				; = 3(p-1)
	COUNT_STOP				; 3
	; last value
	movf	TMR0,w		; test for TMR0 rollover 
	movwf	BFC0		; save 8 bits counter
	subwf	TMR0_old,f
	btfsc	STATUS,Z	; rollover ?
	goto	M_35		; no
	btfsc	STATUS,C
	incf	BFC0-1,f	; 16 bits counter
	btfsc	STATUS,Z
	incf	BFC0-2,f	; 24 bits counter
M_35

	retlw	0x00

;********************************************************************************
; 32 bits number into 7 decimal numbers convertion
; Add or subtract FI or none
;********************************************************************************
BinToDec
;Add or subtract FI or none
MOD_IF
	movf	MODindex,f		; set bit Z
	btfss	STATUS,Z		; MOD = 0 ?
	goto 	MOD1			; no -> next value
	call 	LoadBFCtoACb
	goto	MOD_IF_END		; NO IF
MOD1
	movf	MODindex,w		; LO+IF
	xorlw	0x01			; set bit Z if MOD=1
	btfss	STATUS,Z		; MOD = 1 ?
	goto 	MOD2			; no -> next value
	call 	LoadBFCtoACb	; yes, BFC + IF
	call 	LoadIFbtoACa
	call	Q_add
	goto	MOD_IF_END		
MOD2
	movf	MODindex,w		; LO-IF
	xorlw	0x02			; set bit Z if MOD=2
	btfss	STATUS,Z		; MOD = 2 ?
	goto 	MOD3			; no -> next value
	
	call	CP_LO_IF		; return w = 0 if LO>=IF, else return w = 1
	xorlw	0	
	btfss	STATUS,Z		; Z=1 => LO>=IF
	goto	LO_IF_Error

	call 	LoadBFCtoACb	; yes, BFC - IF
	call 	LoadIFbtoACa
	call	Q_sub
	goto	MOD_IF_END		
MOD3
	movf	MODindex,w		; IF - LO
	xorlw	0x03			; set bit Z if MOD=3
	btfss	STATUS,Z		; MOD = 3 ?
	goto 	MOD4			; no -> next value
	
	call	CP_LO_IF		; return w = 0 if LO>=IF, else return w = 1
	xorlw	1	
	btfss	STATUS,Z		; Z=1 => LO<IF
	goto	LO_IF_Error

	call 	LoadIFbtoACb	; yes, IF - BFC
	call 	LoadBFCtoACa	
	call	Q_sub
	goto	MOD_IF_END		
MOD4
MOD_IF_END
	bsf		Count_OK
	call    Q_b2bcd
	return
LO_IF_Error
	bcf		Count_OK
	return

LoadBFCtoACb
	Movff32	BFC0, ACb0 	; load 32 bits counter into accumulator b
	return
LoadBFCtoACa
	Movff32	BFC0, ACa0 	; load 32 bits counter into accumulator a
	return
LoadIFbtoACa
	Movff32	IFb0, ACa0 	; load 32 bits IF into accumulator a
	return
LoadIFbtoACb
	Movff32	IFb0, ACb0 	; load 32 bits IF into accumulator b
	return

; Compare LO to IF
; LO: mesured local frequency = BFC
; IF: parameter
CP_LO_IF
	movf	IFb3,w		
	subwf	BFC3,w		; BFC3 - IFb3
	btfsc	STATUS,Z	; BFC3 = IFb3 ?
	goto    CP_BFC2_IF2 ; yes, next byte

	btfss	STATUS,C	; BFC3 >= IFb3
	retlw	1			; LO < IF
	retlw	0			; LO > IF

CP_BFC2_IF2
	movf	IFb2,w		
	subwf	BFC2,w		; BFC2 - IFb2
	btfsc	STATUS,Z	; BFC2 = IFb2 ?
	goto    CP_BFC1_IF1 ; yes, next byte

	btfss	STATUS,C	; BFC2 >= IFb2
	retlw	1			; LO < IF
	retlw	0			; LO > IF

CP_BFC1_IF1
	movf	IFb1,w		
	subwf	BFC1,w		; BFC1 - IFb1
	btfsc	STATUS,Z	; BFC1 = IFb1 ?
	goto    CP_BFC0_IF0 ; yes, next byte

	btfss	STATUS,C	; BFC1 >= IFb1
	retlw	1			; LO < IF
	retlw	0			; LO > IF

CP_BFC0_IF0
	movf	IFb0,w		
	subwf	BFC0,w		; BFC0 - IFb0
	btfss	STATUS,C	; BFC0 >= IFb0
	retlw	1			; LO < IF
	retlw	0			; LO >= IF

;********************************************************************************
; Display counter
;********************************************************************************
DisplayCounter
	CURSOR_POSl LCD_LINE0	; Position cursor leftmost on current line

	btfss	Count_OK		; test measure Count_OK
	goto	MeasureError

	LCDDisplay_NumberF BCD9, FORMAT_MHz
	
	movlw	' '	
	call	LCDputChar4		; Display one space 

	LCDDisplay_Textl	MHz_UNIT	; display unit text

	Return
MeasureError
	call	UnderFlowError
	return

UnderFlowError
	LCDDisplay_Textl	UNDERFLOW_ERROR	; display error text
	return
;================================================================================
;								FREQUENCY COUNTER END
;================================================================================

;*****************************************************************************
;                       INITIALISATION PIC        
;*****************************************************************************
PICinit
	BANK1					; passer en banque 1
	movlw	OPTIONVAL		; charger masque
	movwf	OPTION_REG		; initialiser registre option
			; Configurer les PORTS
	movlw	b'00000000'		; masque du PORTB = tout en sortie
	movwf	TRISB			; charger le masque du PORTB
	movlw	b'00000000'		; RA0,RA1 output, RA2 output, RA3, RA4 output 
	movwf	TRISA		 
	BANK0					; passer en banque 0
	clrf	PORTA			; ALL PORT output should output Low.
	bsf		PORTA,RA0		; button SET high
	bsf		PORTA,RA1		; button NEXT high
	clrf	PORTB			; sorties portB  0
	movlw	INTERMASK		; masque interruption
	movwf	INTCON			; charger le masque d'interruption
	BANK1					; passer en banque 1
	movlw	b'11111011'		; RA0,RA1 input, RA2 output, RA3, RA4 input 
	movwf	TRISA		 
	BANK0					; passer en banque 0
	clrf	PROG_FLAGS		; function flag set = 0
	clrf	MODindex		; init default options 
	clrf	IFb0
	clrf	IFb0-1
	clrf	IFb0-2
	clrf	IFb0-3
	return

;================================================================================
;									PROGRAM ROOT 
;================================================================================
start
#ifdef BIN_TO_DEC
	call	BinToDec
#endif

	call	PICinit
	call	LCDinit			; Initialize LCDisplay
	call	LoadOptions 	; Load options from EEPROM
ProgramLoop
	call 	CountFrequency
	call	BinToDec		; binary to decimal conversion
	call	DisplayCounter
	call	isSetup
	goto	ProgramLoop

;================================================================================
;								PROGRAM ROOT END
;================================================================================
	END