;******************************************************************************
; File Name    	: keyer16F628-10.asm
; Version      	: 1.0
; Title:       	: KEYER+USART EEPROM Loader
; Author		: F8EOZ Bernard DECAESTECKER http://f8eoz.com
; MCU			: Written and Tested on Microchip PIC 16F628 Microcontroller
; Compiler     	: Microchip MPASM Assembler V5.51
; IDE          	: Microchip MPLAB IDE v8.92
; Programmer   	: Home made
; Last Updated 	: 22 September 2015
; *******************************************************************
; Hardware:     
;  Tested with PIC16F628 at 4Mhz  XTAL oscillator.       
;  USART use pin 8 as RX (RB1) 
;  USART use pin 9 as TX (RB2) 
; *******************************************************************
; Description: 
; *******************************************************************
; ELoader is built as a Boot Loader
; ELoader is a small piece of firmware, stored in bottom of
; microcontroller's memory, which can write to the EEPROM of 
; that microcontroller. 
; It uses simple generic asynchronous half-duplex communication protocol:
; RX frame from Host:
; 	"L" : Login (26s Timeout at Reset customisable)
; 	"W"aabbccddee : Write Data ccdd into EEPROM @Address aabb ee=cheksum
; 	"P"aabb : Print Data from EEPROM @Address aabb
; 	"X" : Logout and run user program
; TX frame to Host:
; 	"L"ccdd : Login acknowledgment ccdd = EEPROM start address
; 	"K" : Write acknowledgment
; 	"P"ccdd : Data ccdd to print
; 	"!" : Writing data error
; 	"?" : Writing cheksum error
; 	"$" : Writing Boundary error
; PARAMETERS see below:
;   Settle depending on PIC
;   EEPROM_SIZE : Maximum EEPROM bytes
;   FLASH_PROG_SIZE : FLASH PROGRAM bytes
;
;   Choose your Baud Rate with these parameters below:
;	SPBRG_VALUE : Baud Rate
;	SPBRG_BRGH : High or Low speed Baud Rate
;
;   Choose your Login Timeout with these parameters below:
;   ELOADER_TIMER1_OVF  Overflow TIMER1 counter for login timeout
;   ELOADER_TIMER1	Base TIMER1 value
;   ELOADER_T1CKPS  Prescale value
; *******************************************************************
;	KEYER: see keyer12F675
; *******************************************************************
	TITLE "keyer16F628-10.asm"
	errorlevel -302, -306		; no message 302 & 306
	;errorlevel -302: Turn off banking message known tested (good) code
	;errorlevel +302: Enable banking message untested code
	list p=16f628a, r=dec ;Sets the radix for data expressions. The default radix is hex. 	
	#include <p16f628a.inc>

;====================================================================== 	
;====================================================================== 	
; USER PROGRAMM DEFINES
;====================================================================== 	
;====================================================================== 	

; Power on Timer, no Watchdog, HS-Oscillator, no Brown out, no Low Volt programming
	__CONFIG	_PWRTE_ON & _WDT_OFF & _HS_OSC & _BODEN_OFF & _LVP_OFF
;-----------------------------------------------------------------------------
;	Release
#define	V1_0
;-----------------------------------------------------------------------------
;	TEST and DEBUG
;	Comment out the next line [;#define TEST] if no test and debug
;#define	__TEST	; TEST and DEBUG mode
	ifdef	__TEST
		MESSG	"TEST and DEBUG = __TEST"
	endif
;#define	__TRON			; Buzzer TRACE ON (old programmer mania!)
;#define	__ADRES_TRON	; ADC result TRACE ON 
;#define	__ADRES_TRON1	; ADC result TRACE ON 
	ifdef	__TRON
		MESSG	"TEST and DEBUG = __TRON"
	endif
	ifdef	__ADRES_TRON
		MESSG	"TEST and DEBUG = __ADRES_TRON"
	endif
;-----------------------------------------------------------------------------
;	Programm and Data memory
;#define RESET_VECTOR		0x0000	; Address of RESET Vector
#define ISR_VECTOR			0x0004	; Address of Interrupt Vector
#define RAM_BANK0			0x0020	; 80 General Purpose registers (SRAM)
#define RAM_BANK1			0x00A0	; 80 General Purpose registers (SRAM)
#define COMMON_RAM_DATA		0x0070	; 16 General Purpose registers (SRAM)
#define MSG_DATA			0x03EB
;#define OSCCAL_FACTORY_ADR 	0x03FF
#define EEPROM_DATA			0x2100	; EEPROM address 
#define EEPROM_DATA_MAX		0x7F	; EEPROM 128 bytes 

;-----------------------------------------------------------------------------
; Input Output Port
;       GP0 - Analog input from keypad
;       GP1 - dah paddle
;       GP2 - dit paddle
;       GP3 - Reset button between ground and MCLR.  MCLR tied high via 10k resister.
;       GP4 - Drive Tx
;       GP5 - Drive Buzzer 
#define AN_KPAD 	PORTA,RA0	; (AN0) keypad input
#define GP_DAH 		RB5			; (GP1) DAH paddle
#define GP_DIT 		RB0			; (GP2) DIT paddle
#define GP_TX 		RB6			; Tx output
#define GP_BUZ 		RB4			; Buzzer output
#define GPIO		PORTB
#define	PORT_BUZ	PORTB
;----- UCON Bits -----------------------------------------------------
IAMBIC_KEY 	EQU 0x00	; StraightKey=0 Iambic= 1
BUZE	 	EQU 0x01	; = 1 if side tone buzzer enable
TXF 		EQU 0x02	; Tx flag = 1 if message sending to Tx
MSGF		EQU 0x03	; msg flag = 1 if send message running 
KPADF		EQU 0x04	; = 1 if keypad unlocked
MOD_B		EQU 0x05	; = 1 if keyer mode B, 0 mode A
BUZF		EQU 0x06	; buzzer flag = 1 if message sending to buzzer

;----- DEFAULT Options --------------------------------------------------
; Iambic Key + Buzzer on
#define UCON_DEFAULT (1<<IAMBIC_KEY) | (1<<BUZE) | (1<<BUZF)

;----- UCON1 Bits -----------------------------------------------------
; Compare Status Register
A_GT_B 		EQU 0x00	; = 1 if A>B
A_EQ_B 		EQU 0x01	; = 1 if A=B
A_LT_B 		EQU 0x02	; = 1 if A<B
; Paddle status
BOTHPADF	EQU 0x04	; = 1 if both paddles squeezed
IAMBIC_SW 	EQU 0x05	; Toggle iambic switch: 1=DIT	0=DAH

;-----------------------------------------------------------------------------
; MORSE
;-----------------------------------------------------------------------------
; Spacing and timing

; To standardize the International Code Transmission Speed, 
; the 5-letter word PARIS or CODEX is used to establish the number of words-per-minute unit 
; For example, if the word PARIS was sent 5 times in a minute, 
; the transmission speed would be 5-words-per-minute or WPM.
; The following relationships exist between the elements of the code (dits and dahs), 
; the characters (letters) and the words:
; The DIT is the Basic UNIT of Length.
; The DAH is equal in length to three DITS.
; The space between the DITS and DAHS within a character (letter) is equal to one DIT.
; The space between characters (letters) in a word is equal to three DITS.
; The space between words is equal to seven DITS.
; The following formula calculates the dot period in milliseconds  from
; the Morse code speed in words per minute:
; dot period = ( 1200 / speed )

; This formula arises from the use of the word PARIS as a standard word
; for calibrating Morse code speed.  PARIS is 50 units long when sent in
; Morse code. Analysis of English plain-text indicates that the average
; word is 50 units, including spaces.

;---------------------------------------------------------------------- 	
; Morse constant
#define X_TIME_5US 199		; time multiplier 	
#define DU 60 ;dot unit CODEX= 7 dots + (10 sign-gap * 1 dot) + (8 dashes * 3 dots) + (4 code-gap * 3 dots) + (1 word-gap * 7 dots)
;#define DU 50 ; dot unit PARIS= 10 dots + (9 sign-gap * 1 dot) + (4 dashes * 3 dots) + (4 code-gap * 3 dots) + (1 word-gap * 7 dots)
#define WG 7 ;  word-gap in dot unit

;---------------------------------------------------------------------- 	
; Morse code
;
; Letters A, B, C... Z = 1 to 4 bit coded character 
; xxxx: high nibble = Morse code; 1=DAH, 0=DIT,...
; yyyy: low nibble = 4 bit Morse code Frame vector. 1=DAH or DIT, 0=end
;             xxxxyyyy
cA EQU   	b'01001100'    ; A = .- 	(2 bit)
cB EQU   	b'10001111'    ; B = -... 	(4 bit)
cC EQU   	b'10101111'    ; C = -.-. 	(4 bit)
cD EQU    	b'10001110'    ; D = -.. 	(3 bit)
cE EQU    	b'00001000'    ; E = . 		(1 bit)
cF EQU    	b'00101111'    ; F = ..-.	(4 bit)
cG EQU   	b'11001110'    ; G = --.  	(3 bit)
cH EQU    	b'00001111'    ; H = ....	(4 bit)
cI EQU    	b'00001100'    ; I = ..		(2 bit)
cJ EQU    	b'01111111'    ; J = .---	(4 bit)
cK EQU    	b'10101110'    ; K = -.-  	(3 bit)
cL EQU    	b'01001111'    ; L = .-.. 	(4 bit)
cM EQU   	b'11001100'    ; M = --   	(2 bit)
cN EQU   	b'10001100'    ; M = -.   	(2 bit)
cO EQU   	b'11101110'    ; O = ---  	(3 bit)
cP EQU    	b'01101111'    ; P = .--.	(4 bit)
cQ EQU   	b'11011111'    ; Q = --.- 	(4 bit)
cR EQU    	b'01001110'    ; R = -.. 	(3 bit)
cS EQU   	b'00001110'    ; S = ...  	(3 bit)
cT EQU    	b'10001000'    ; T = - 		(1 bit)
cU EQU    	b'00101110'    ; U = ..-	(3 bit)
cV EQU    	b'00011111'    ; V = ...-	(4 bit)
cW EQU    	b'01101110'    ; W = .--  	(3 bit)
cX EQU   	b'10011111'    ; X = -..- 	(4 bit)
cY EQU   	b'10111111'    ; Y = -.-- 	(4 bit)
cZ EQU   	b'11001111'    ; Z = --.. 	(4 bit)

;---------------------------------------------------------------------- 	
; Morse code
; Numbers, Punctuation, Prosigns (Procedural signals)
; 8 bit coded character
; 3 bits b2...b0 = 010 = 5 bits format
; 3 bits b2...b0 = 011 = special format
; 2 bits b1...b0 = 01 = 6 bits format
; morse code; 1=DAH, 0=DIT,...
;
;
; Numbers 5 bit format
; 
;             xxxxx010 		-> Frame = 11111000
c1 	EQU  	b'01111010'    ; 1 = .---- 
c2 	EQU  	b'00111010'    ; 2 = .---- 
c3 	EQU  	b'00011010'    ; 3 = .---- 
c4 	EQU  	b'00001010'    ; 4 = .---- 
c5 	EQU  	b'00000010'    ; 5 = .---- 
c6 	EQU  	b'10000010'    ; 6 = -.... 
c7 	EQU  	b'11000010'    ; 7 = --... 
c8 	EQU  	b'11100010'    ; 8 = ---.. 
c9 	EQU  	b'11110010'    ; 9 = ----. 
c0 	EQU  	b'11111010'    ; 0 = ----- 

; Punctuation 5 bit format
;             		  xxxxx010		-> Frame = 11111000
cSlashBar 	EQU  	b'10010010'    ; / = -- 
cParentOpen	EQU  	b'10110010'    ; ( = ---
cAmpers		EQU  	b'01000010'    ; & = -
cEqual		EQU  	b'10001010'    ; = = -- 
cPlus 		EQU  	b'01010010'    ; + = .-.-.

; Punctuation 6 bit format
;             		  xxxxxx01		-> Frame = 11111100
cPer		EQU  	b'01010101'    ; . = ---
cComa		EQU  	b'11001101'    ; , = ----
cQM			EQU  	b'00110001'    ; ? = --
cQuot		EQU  	b'01111001'    ; ' = ----
cExcl		EQU  	b'10101101'    ; ! = ----
cParentClose	EQU  	b'10110101'    ; ) = ----
cCol		EQU  	b'11100001'    ; : = ---
cSCol		EQU  	b'10101001'    ; ; = --- 
cMinus		EQU  	b'10000101'    ; - = --
cUndS		EQU  	b'00110101'    ; _ = ---
cQuotM		EQU  	b'01001001'    ; " = --

; Prosigns special format
;             		  xxxxx011		-> code >127 Frame = 011
_WG 		EQU  	b'10000011'    ; WORD GAP = 2 dit time delay
_CS 		EQU  	b'10001011'    ; Alias for Call sign
_TUNEC	 	EQU  	b'10010011'    ; Tune mode constant carrier 
_TUNEP 		EQU  	b'10011011'    ; Tune mode 50% duty cycle dits
_ERR 		EQU  	b'10100011'    ; Error
_ERR8b		EQU  	b'00000000'    ; Error= ........ 8 bit format substitute
_SK 		EQU  	b'10101011'    ; SK
_SK6b 		EQU  	b'00010101'    ; SK = ...-.-  6 bit format substitute
_AR		 	EQU  	b'10110011'    ; AR
_AR5b	 	EQU  	b'01010010'    ; AR = .-.-.   5 bit format substitute
_BT		 	EQU  	b'10111011'    ; BT
_BT5b	 	EQU  	b'10001010'    ; BT = --	  5 bit format substitute
;
;----------------------------
; Header of message directive
_HOM		EQU     0x02
; End of message part
_EOP 		EQU		0x01
; End of message
_EOM 		EQU		0x00 

;-------
; KEYPAD 
;-------
; Definition
KPAD_00	EQU 0
KPAD_10	EQU 1
KPAD_20	EQU 2
KPAD_01	EQU 3
KPAD_11	EQU 4
KPAD_21	EQU 5
KPAD_02	EQU 6
KPAD_12	EQU 7
KPAD_22	EQU 8
KPAD_FF	EQU 0xFF ; no key
; Assign key 3x3 push button
; This allows one to change keypad layout without change algorithm
; My Layout:
; <Send0> <Spdup> <TuneC> (y2)
; <Send1> <Sdown> <TuneP> (y1)
; <Send2> <Buzz.> <ModAB> (y0)
;  (x2)	   (x1)	   (x0)
KPAD_SEND0		EQU KPAD_22
KPAD_SEND1		EQU KPAD_21
KPAD_SEND2	 	EQU KPAD_20
KPAD_SPEED_UP	EQU KPAD_12
KPAD_SLOW_DOWN 	EQU KPAD_11
KPAD_BUZ	 	EQU KPAD_10
KPAD_TUNEC		EQU KPAD_02
KPAD_TUNEP		EQU KPAD_01
KPAD_MOD		EQU KPAD_00

; Keypad maximum voltage threshold: analog to digital convert
; Ra = colomn resistor
; Rb = row resistor
; 3-by-3 keypad is a matrix x,y of resistor network divided into two row and column sections.
;
; Vref
;  |
; R(x,y)
;  |----->V(x,y)
; Rb
;  |
; Vss
;
; V(x,y) = Vref * ( Rb /(xRa + yRb + Rb)) (1)
; R(x,y) = xRa + yRb
; let Rb=nRa (n>2 to avoid duplicate value)
; equation (1) becomes V(x,y) = Vref * ( nRa /(xRa + nyRa + nRa)) = Vref * n /(x+ny+n) (2)
; let n= 3, equation (2) becomes V(x,y) = Vref * 3 /(x+3y+3) (3)

; Fitted after real analyze

;#define VTOLPLUS 10 ; tolerance
;#define VTOLMINUS 1 ; tolerance

#define KPAD_00_MAX (1016+16);key 0,0 (maximum maximorum)
#define KPAD_00_MIN (1016-4);key 0,0 
#define KPAD_10_MAX (757+5);key 1,0 
#define KPAD_10_MIN (757-5);key 1,0 
#define KPAD_20_MAX (605+5);key 2,0
#define KPAD_20_MIN (605-5);key 2,0 
; Row 1
#define KPAD_01_MAX (502+6)	;key 0,1
#define KPAD_01_MIN (502-6)	;key 0,1 
#define KPAD_11_MAX (438+5);key 1,1 
#define KPAD_11_MIN (438-5);key 1,1 
#define KPAD_21_MAX (385+5)	;key 2,1
#define KPAD_21_MIN (385-5)	;key 2,1 
; Row 2
#define KPAD_02_MAX (345+5)	;key 0,2
#define KPAD_02_MIN (345-5)	;key 0,2 
#define KPAD_12_MAX (308+7)	;key 1,2 
#define KPAD_12_MIN (308-8)	;key 1,2 
#define KPAD_22_MAX (287+5)	;key 2,2
#define KPAD_22_MIN (287-5)	;key 2,2 


; NO KEY
#define KPAD_FF_MAX 256	; no key 
#define KPAD_FF_MIN 0	; no key 

;---------------------------------------------------------------------- 	
;#define __TEST
	ifdef	__TEST
		#define	ISR_TMR0_OVF 1	; timer0 overflow 
	else  			
		#define	ISR_TMR0_OVF 11	; timer0 overflow 
	endif

; @4MHz instruction cycle = 1us
; TIMER1 prescaler = 1:8
; 16 bits TIMER1 = (256*256) = 65536
; 0.5 second = 500000us 
; init value = 65536 - (500000/8) = 3036 = 0xBDC
	ifdef	__TEST
		#define TMR1_INIT  65536 - 10 ; if test
		#define	TMR1_OVF_INIT 1	; timer0 overflow 
	else  			
		#define TMR1_INIT  (256*256) - (500000/8)
		#define TMR1_OVF_INIT  2 ; 2 overflow = 2*0.5 = 1s
	endif
 
;---------------------------------------------------------------------- 	
; ADC Config
#define ADC_CON b'10000001'
#define ADC_AN0 b'00000000'
#define ADC_AN1 b'00000100'
#define ADC_AN2 b'00001000'
#define ADC_AN3 b'00001100'
;         		  1-------  ADFM: 1=right justified result
;         		  -0------  VCFG: 0=Vdd is voltage reference
;         		  --xx----  not implemented
;         		  ----00--  00=select channel 00 (AN0=GP0)
;				      01 	= Channel 01 (AN1=GP1) 	
;				      10	= Channel 02 (AN2=GP2) 	
;					  11 	= Channel 03 (AN3=GP4)	
;         		  ------0-  GO/NOT DONE Status bit: 0=A/D conversion not started
;         		  -------1  ADON: Conversion Status bit 1=A/D converter module is operating

#define ADC_AN_KPAD ADC_AN0 ; assign Keypad to Channel 00

;-----------------------------------------------------------------------------
; PIC MACRO
;-----------------------------------------------------------------------------
BANK0	macro
		bcf	STATUS,RP0		
		endm

BANK1	macro
		bsf	STATUS,RP0		
		endm

;---------------------------------------------------------------------- 	
; DECODER MACRO
;---------------------------------------------------------------------- 	
; encoder: Input from a RAM message table
; input: _TAB_ADR_ = table address
; input: _DATA_ADR_ = data line address into table
; output WREG = relative address
SEND_SERVICE_MSG macro _TAB_ADR_, _DATA_ADR_
	bsf		UCON,BUZF		; Side tone buzzer on
	bcf		UCON,TXF		; unset flag Tx
	movlw	_DATA_ADR_ - _TAB_ADR_ - 1 - 3
	call 	Encoder_RAMinp	; Send text @WREG
	endm


;====================================================================== 	
;====================================================================== 	
; END USER PROGRAMM DEFINES
;====================================================================== 	
;====================================================================== 	

;---------------------------------------------------------------------- 	
; EEPROM LOADER DEFINES 
;---------------------------------------------------------------------- 	

;------------------------- 	
;	Start Programm Address
#define RESET_VECTOR 0x0000	; Address of RESET Vector

;------------------------- 	
; Memory PIC features
#DEFINE FLASH_PROG_SIZE 0x7FF ; 2K FLASH PROGRAM
#DEFINE EEPROM_SIZE H'7F' ; EEPROM = 128 bytes

;------------------------- 	
; Test and Debug
#DEFINE ELOADER_TEST 0

;-----------------------------------
; SPRG: BAUD RATE GENERATOR REGISTER
; FOSC  = 4MHz XTAL
; SPBRG = (FOSC/(Desired Baud Rate*16))  1	
; SPBRG = (4000000/(9600*16))  1 = 25,04 = 25
; Calculated Baud Rate = 4000000/(16 * (25+1)) = 9615,38 bauds
; Error = (9615,38  9600) / 9600 = 0,16% <	4%
SPBRG_VALUE = D'25'
#define SPBRG_BRGH 1

;--------------------------------------
; LOGIN TIMEOUT PARAMETERS
; Instruction cycle = FOSC/4 = 1s @4MHz
; 65536 * 2 * 200 = 26214400us = 26s
; 65536 = TIMER1 0 to 65535
; 2 = prescale value
; 200 = roll over TIMER1
#define ELOADER_TIMER1_OVF  D'200'
;#define ELOADER_TIMER1_OVF  D'001'
ELOADER_TIMER1	EQU	D'65536'
;ELOADER_TIMER1	EQU	D'65534'
#define ELOADER_T1CKPS H'10' ; Prescale=1:2
;#define ELOADER_T1CKPS H'00' ; Prescale=1:1
;#define ELOADER_T1CKPS H'20' ; Prescale=1:4
;#define ELOADER_T1CKPS H'30' ; Prescale=1:8

	IF ELOADER_TIMER1_OVF == 0
		ERROR "ELOADER_TIMER1_OVF Overflow time parameter must be > 0"
	ENDIF

#define ELoaderSize	0xA2 ; EEPROM Loader size
;#define ELoader	ELoaderExit+5		     ; adresse de base du bootloader
;#define ELoaderAdrEnd	FLASH_PROG_SIZE 
#define ELoaderRoot	(FLASH_PROG_SIZE)-ELoaderSize+1 ; EEPROM Loader Root address
;#define NumRetries	1			     ; number of writing retries
; Communication Query Code
#define ELOADER_WRITE	    		"W" ; Query Write into RAM			 
#define ELOADER_WRITE_EEPROM		"E" ; Query Write into EEPROM			 
#define ELOADER_PRINT_EEPROM		"P" ; Query Print EEPROM			 
#define ELOADER_WRITE_ACK	   		"K" ; Write OK
#define ELOADER_WRITE_ERR	    	"!" ; Write KO data error
#define ELOADER_WRITE_BOUNDARY_ERR	"$" ; Write KO boundary error
#define ELOADER_CHEKSUM_ERR    		"?" ; Write KO Cheksum rror
#define ELOADER_LOGIN				"L" ; Query Login to device
#define ELOADER_LOGIN_ACK  			"L" ; Connect Login to device Acknowledge
#define ELOADER_LOGOUT				"X" ; Logout and Run Program
#define NEW_LINE 					0x0A ; End of a TX line

;-----------------------------------------------------------------------------
; EEPROM LOADER FILES REGISTER
; This data can be redefined by program
#define ELOADER_CBLOCK 0x70 ; Common RAM

; Query BUFFER 6 Bytes
ELoaderBuffer   EQU ELOADER_CBLOCK
ELoaderFiller0  EQU ELOADER_CBLOCK
ELoaderFiller1	EQU ELOADER_CBLOCK+1
ELoaderFiller2	EQU ELOADER_CBLOCK+2
ELoaderFiller3	EQU ELOADER_CBLOCK+3
ELoaderFiller4	EQU ELOADER_CBLOCK+4
ELoaderFiller5	EQU ELOADER_CBLOCK+5
; Query LOGIN Refefines BUFFER: "L"
; Query LOGOUT Refefines BUFFER: "X"
; Query WRITE EEPROM Redefines BUFFER: "W"aabbccddee
; Query PRINT EEPROM Redefines BUFFER: "P"aabb
ELoaderQcode    EQU ELOADER_CBLOCK	 ;Query Code
ELoaderAddrH    EQU ELOADER_CBLOCK+1 ;aa
ELoaderAddrL    EQU ELOADER_CBLOCK+2 ;bb
ELoaderDataH    EQU ELOADER_CBLOCK+3 ;cc
ELoaderDataL    EQU ELOADER_CBLOCK+4 ;dd
ELoaderChkSum   EQU ELOADER_CBLOCK+5 ;ee

ELoaderOVFtime	EQU	ELOADER_CBLOCK+9

;---------------------------------------------------------------------- 	
; END EEPROM LOADER DEFINES 
;---------------------------------------------------------------------- 	

;====================================================================== 	
;====================================================================== 	
; USER RAM FILES
;====================================================================== 	
;====================================================================== 	

;---------------------------------------------------------------------- 	
	cblock RAM_BANK0
;---------------------------------------------------------------------- 	

	;--------------- 	
	; Register Files
	UCON:1		; 8 bits User Program Control Register
	UCON1:1		; 8 bits User Program Control Register 

	;-------------- 	
	; Counter Files 
	DelayCount:1		; delay counter

	;------------ 	
	; Timer Files
	TimeCount1:1        ; 8 bits counter
	TimeCount2:1        ; 8 bits counter
	WordPerMn:1			; 8 bits word per mn counter
	WordPerMnMax:1		; 8 bits maximum word per mn
	UnitPeriod:1     	; 8 bits current Unit Period counter
	SpeedPtr:1			; Speed scale table pointer

	;-------------- 	
	;Encoder Files
	WGapCount:1      	; 8 bits counter
	MorseCode:1			; 8 bits current Morse code
	ControlTape:1		; 8 bits current Morse code Frame vector
	BitFramePtr:1
	; Morse message Table
	TabPointer:1		; Table pointer 
	TabPointerSave:1	; Table pointer save for 2nd level
	;-------------------- 	
	; Key Interface Files
	GPIOcopy:1			; 8 bits current GPIO

	;----------------------- 	
	; Keypad Interface Files
	KeyPadXY:1     		; 8 bits current x,y key
	KeyPadXYPre:1     	; 8 bits previous x,y key
	KeyPadXYtemp:1     	; 8 bits temporary x,y key
	; ADC
	ADCVthPointer:1		; Table Voltage threshold pointer
	; Send message
	MsgN:1				; Message #
	BeaconTime:1		; Beacon Time of Message #

	;----------------------------- 	
	; Compare subroutine A?B Files
	CmpAL:1				; A LOW byte value
	CmpAH:1				; A HIGH byte value
	CmpBL:1				; B LOW byte value
	CmpBH:1				; B HIGH byte value
	
	;------------------------------- 	
	; math.sub Math subroutine Files
	; 16 bytes reseved
	AC1:1       		; 16-bit Accumulator a, lsb+1
	AC0:1       		; 16-bit Accumulator a, ls-byte

	;---------------- 	
	; Interrupt Files
	ISRwreg:1			; save WREG
	ISRstatus:1			; save STATUS
	TMR1ovf:1			; Timer1 overflow

	;------------- 	
	; EEPROM Files
	EEByteCounter:1	; EEPROM byte counter
	EEMessagePtr:1	; EEPROM 1st message start address
	endc	; End of General Purpose registers

	cblock COMMON_RAM_DATA

	;------------------
	; Delta Sigma Files                     
	counter:2
	ADRESL:1
	ADRESH:1
	SampleCount:1
	endc				; End of General Purpose registers                     

;---------------------------------------------------------------------- 	
	cblock RAM_BANK1
;---------------------------------------------------------------------- 	
	endc				; End of General Purpose registers                     
;====================================================================== 	
;====================================================================== 	
; END USER RAM FILES
;====================================================================== 	
;====================================================================== 	

	ORG		RESET_VECTOR  ; RESET Vector
	nop				      
	pagesel ELoaderStart
	goto	ELoaderStart
	org		ISR_VECTOR    ; interrupt vector
	goto	ISR

;====================================================================== 	
;====================================================================== 	
; USER PROGRAM ROOT 
;====================================================================== 	
;====================================================================== 	

;---------------------------------------------------------------------- 	
; Timer: Speed Control: Speed scale Table
; input: SpeedFieldPtr = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
BIT_FRAME_TABLE
	movlw 	HIGH(BIT_FRAME); PCLATH = HIGH bit address
	movwf 	PCLATH 				; 		
	movf	BitFramePtr,W		; for PCL
	addwf	PCL ,F 				; Jump to character pointed to in W register
BIT_FRAME ; sort ascending by ASCII
#define ASCII_MIN " "
	DT  _WG ; Space ==>FIRST CODE 0x20 DON'T REMOVE
	DT  cExcl ; !
	DT  cQuotM ; "
	DT  _WG ; # not used
	DT  _WG ; $ not used
	DT  _WG ; % not used
	DT  cAmpers ; &
	DT  cQuot ; '
	DT  cParentOpen ; (
	DT  cParentClose ; )
	DT  _WG ; *
	DT  cPlus ; +
	DT  cComa ; ' not used
	DT  cMinus ; - (hyphen)
	DT  cPer ; . (dot)
	DT  cSlashBar ; /
	DT  c0
	DT  c1
	DT  c2
	DT  c3
	DT  c4
	DT  c5
	DT  c6
	DT  c7	
	DT  c8
	DT  c9
	DT  cCol ; :
	DT  cSCol ; ;
	DT  _WG ; < not used
	DT  cEqual ; =
	DT  _WG ; > not used
	DT  cQM ; ?
#define	CS_ALIAS "@"
	DT  _CS ;@
	DT	cA
	DT  cB
	DT  cC	
	DT  cD	
	DT	cE		
	DT	cF		
	DT  cG
	DT  cH
	DT  cI
	DT  cJ
	DT  cK
	DT  cL
	DT  cM
	DT  cN	
	DT  cO	
	DT  cP	
	DT  cQ	
	DT  cR	
	DT  cS	
	DT  cT	
	DT  cU	
	DT  cV
	DT  cW
	DT  cX	
	DT  cY	
	DT  cZ  ; ==>LAST CODE 0x5A DON'T REMOVE
#define ASCII_MAX "Z"
	IF ((HIGH ($)) != (HIGH (SPEED_SCALE)))
    	ERROR "BIT_FRAME CROSSES PAGE BOUNDARY!"
	ENDIF;

;---------------------------------------------------------------------- 	
; ADC Threshold Voltage Table
; ADC is 10 bits, and has a max. value of 0x3FF or 1023.
; With a 5 V reference, this gives a resolution of 5/1024
; or 4.88 mV.
; input: ADCVthPointer = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
ADC_Vth_TABLE
	movlw 	HIGH(ADC_Vth_KEYPAD); PCLATH = HIGH bit address
	movwf 	PCLATH 				; 		
	movf	ADCVthPointer,W		; for PCL
	addwf	PCL ,F ; Jump to character pointed to in W register
ADC_Vth_KEYPAD ; maximum voltage threshold for keypad: analog to digital convert 

	; real digit value saved into EEPROM for analyze (use __ADRES_TRON to get result)
	; 	(2)   (1)   (0)
	; 8---- 7---- 6---- (2)
	; 01 18 01 34 01 58 
	; 01 1F ** ** 01 59 DeltaSigma ** ** = 01 30, 01 39, 01 3A
	; 5---- 4---- 3---- (1)
	; 01 82 01 B8 02 01
	; 01 81 01 D6 01 F6 DeltaSigma 
	; 2---- 1---- 0---- (0)
	; 02 69 03 01 03 FF
	; 02 5D 02 F5 03 F8 DeltaSigma 
    
	; -------xy: digit maximum voltage threshold 	: digit minimum voltage threshold 
	DT	KPAD_FF,(HIGH KPAD_FF_MAX),LOW (KPAD_FF_MAX),(HIGH KPAD_FF_MIN),LOW (KPAD_FF_MIN) 	
	DT	KPAD_22,(HIGH KPAD_22_MAX),LOW (KPAD_22_MAX),(HIGH KPAD_22_MIN),LOW (KPAD_22_MIN) 	
	DT	KPAD_12,(HIGH KPAD_12_MAX),LOW (KPAD_12_MAX),(HIGH KPAD_12_MIN),LOW (KPAD_12_MIN) 	
	DT	KPAD_02,(HIGH KPAD_02_MAX),LOW (KPAD_02_MAX),(HIGH KPAD_02_MIN),LOW (KPAD_02_MIN) 	
	DT	KPAD_21,(HIGH KPAD_21_MAX),LOW (KPAD_21_MAX),(HIGH KPAD_21_MIN),LOW (KPAD_21_MIN) 	
	DT	KPAD_11,(HIGH KPAD_11_MAX),LOW (KPAD_11_MAX),(HIGH KPAD_11_MIN),LOW (KPAD_11_MIN) 	
	DT	KPAD_01,(HIGH KPAD_01_MAX),LOW (KPAD_01_MAX),(HIGH KPAD_01_MIN),LOW (KPAD_01_MIN) 	
	DT	KPAD_20,(HIGH KPAD_20_MAX),LOW (KPAD_20_MAX),(HIGH KPAD_20_MIN),LOW (KPAD_20_MIN) 	
	DT	KPAD_10,(HIGH KPAD_10_MAX),LOW (KPAD_10_MAX),(HIGH KPAD_10_MIN),LOW (KPAD_10_MIN) 	
	DT	KPAD_00,(HIGH KPAD_00_MAX),LOW (KPAD_00_MAX),(HIGH KPAD_00_MIN),LOW (KPAD_00_MIN) 	; 5V
	; don't remove line below
	retlw 0xFF	; high value = end of table
	IF ((HIGH ($)) != (HIGH (ADC_Vth_KEYPAD)))
    	ERROR "ADC_Vth_KEYPAD CROSSES PAGE BOUNDARY!"
	ENDIF;
 
;---------------------------------------------------------------------- 	
; Timer: Speed Control: Speed scale Table
; input: SpeedFieldPtr = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
SPEED_SCALE_TABLE
	movlw 	HIGH(SPEED_SCALE); PCLATH = HIGH bit address
	movwf 	PCLATH 				; 		
	movf	SpeedPtr,W			; for PCL
	addwf	PCL ,F 				; Jump to character pointed to in W register
SPEED_SCALE ; speed and UnitPeriod
#define SPEED_MAX 50
#define SPEED_DEFAULT 12
;	   WPM:UnitPeriod--------
	DT	50,60000/((DU*50)-WG) ; 50 wpm max
	DT  49,60000/((DU*49)-WG)	
	DT  48,60000/((DU*48)-WG)	
	DT  47,60000/((DU*47)-WG)	
	DT	46,60000/((DU*46)-WG)		
	DT	45,60000/((DU*45)-WG)		
	DT  44,60000/((DU*44)-WG)
	DT  43,60000/((DU*43)-WG)
	DT  42,60000/((DU*42)-WG)
	DT  41,60000/((DU*41)-WG)
	DT  40,60000/((DU*40)-WG)
	DT  39,60000/((DU*39)-WG)
	DT  38,60000/((DU*38)-WG)
	DT  37,60000/((DU*37)-WG)	
	DT  36,60000/((DU*36)-WG)	
	DT  35,60000/((DU*35)-WG)	
	DT  34,60000/((DU*34)-WG)	
	DT  33,60000/((DU*33)-WG)	
	DT  32,60000/((DU*32)-WG)	
	DT  31,60000/((DU*31)-WG)	
	DT  30,60000/((DU*30)-WG)	
	DT  29,60000/((DU*29)-WG)
	DT  28,60000/((DU*28)-WG)
	DT  27,60000/((DU*27)-WG)	
	DT  26,60000/((DU*26)-WG)	
	DT  25,60000/((DU*25)-WG)
	DT  24,60000/((DU*24)-WG)	
	DT  23,60000/((DU*23)-WG)
	DT  22,60000/((DU*22)-WG)
	DT  21,60000/((DU*21)-WG)
	DT  20,60000/((DU*20)-WG)
	DT  19,60000/((DU*19)-WG)
	DT  18,60000/((DU*18)-WG)	
	DT  17,60000/((DU*17)-WG)
	DT  16,60000/((DU*16)-WG)
	DT  15,60000/((DU*15)-WG)	
	DT  14,60000/((DU*14)-WG)
	DT  13,60000/((DU*13)-WG)	
	DT  12,60000/((DU*12)-WG)
	DT  11,60000/((DU*11)-WG)
	DT  10,60000/((DU*10)-WG)
	DT  09,60000/((DU*09)-WG)	
	DT  08,60000/((DU*08)-WG)
	DT  07,60000/((DU*07)-WG)	
	DT  06,60000/((DU*06)-WG)
#define SPEED_MIN 50
	DT  05,60000/((DU*05)-WG) ; 5 wpm min
	; don't remove line below
	retlw 0	; low value = end of table
	IF ((HIGH ($)) != (HIGH (SPEED_SCALE)))
    	ERROR "SPEED_SCALE CROSSES PAGE BOUNDARY!"
	ENDIF;
;------------------------- 	
; USER PROGRAM SUBROUTINES 
;------------------------- 	

;---------------------------------------------------------------------- 	
; Timer: Wait for one Unit Period long 	
; @4MHz one basic instruction duration = 1us 	
; @20WPM one Unit Period long UnitPeriod = 60 ms 	
; Routine wait for: 60 * (2us+(199 * 5us)+3us)= 60ms (+ 2us neglected) 	
; Input arg = UnitPeriod	
TimerUnitP 	
  	movfw	UnitPeriod		; Time Unit is argument (1us neglected)
  	movwf   TimeCount1      ; Counter for UnitPeriod * 1ms (1us neglected)	
; CALL for 1ms: set TimeCount1 before calling
; 1ms = 2us+(199 * 5us)+3us
TimerUnitP_1ms
  	movlw   X_TIME_5US			; 1 us
  	movwf   TimeCount2       	; 1 us
TimerUnitP_5us
  	nop                       	; 1 us to give 5 us inner loop 	
  	nop                       	; 1 us 	
  	decfsz  TimeCount2,f      	; 1 us if no skip
								; 2 us if skip	
  	goto    TimerUnitP_5us  	; 2 us 	
  
	decfsz  TimeCount1,f		; 1 us if no skip
								; 2 us if skip	
  	goto    TimerUnitP_1ms		; 2 us
  	return						; 2us neglected

;---------------------------------------------------------------------- 	
; Delay: Wait for (DelayCount * 5us) + 1us + 2us
; Input: WREG= factor DelayCount
Delay
  	movwf   DelayCount       	; 1 us
Delay_5us
  	nop                       	; 1 us to give 5 us inner loop 	
  	nop                       	; 1 us 	
  	decfsz  DelayCount,f      	; 1 us if no skip
								; 2 us if skip	
  	goto    Delay_5us		  	; 2 us 	
  	return						; 2us

;---------------------------------------------------------------------- 	
; Get speed from table 
; Input: WREG = Word Per Mn 
; Output: UnitPeriod
getUnitPeriod
	movwf	SpeedPtr
	addwf	SpeedPtr,W
	sublw	SPEED_SCALE - SPEED_SCALE_TABLE - 1 - 3 + SPEED_MAX + SPEED_MAX + 1	; W = value relative address 
	;movwf	SpeedItemPtr 		; Ptr to WPM
	;movwf	SpeedFieldPtr
	;incf	SpeedFieldPtr 		; Ptr to UnitP
	movwf	SpeedPtr			; Ptr to UnitPeriod
	call	SPEED_SCALE_TABLE
	movwf	UnitPeriod			; Load current Morse UnitPeriod
	return

;---------------------------------------------------------------------- 	
; Get BitFrame from table 
; Input: WREG = ASCII 
; Output: WREG
getBitFrame
	movwf	BitFramePtr
	; is in table limit?
	sublw	ASCII_MAX ; W = h5A - W
	btfss	STATUS,C
	goto	getBitFrame_Out ; W<0: C=0
	movfw	BitFramePtr
	sublw	ASCII_MIN - 1 ; W = h19 - W
	btfsc	STATUS,C
	goto	getBitFrame_Out ; W>=0: C=1
	; yes: lookup table
	movlw	BIT_FRAME - BIT_FRAME_TABLE - 1 - 3 + ASCII_MIN	; W = value relative address "+"
	subwf	BitFramePtr,f
	call	BIT_FRAME_TABLE
	return
getBitFrame_Out
	movfw	BitFramePtr ; restore code
	return

;---------------------------------------------------------------------- 	
; Encoder: Input from EEPROM 
Encoder_EEPROMinp
	;movlw	LOW(EE_MESSAGE) - 1	; msg start address
	BANK0
	decf	EEMessagePtr,W	; msg start address -1
	BANK1
	movwf	EEADR	
	call	Encoder_EEPROMinp_GetHead
	xorlw	-1				; Check if error not found
	btfsc	STATUS, Z		; Z=1 returned if error not found
	goto	Encoder_EEPROMinp_Error
	BANK1
	movfw	EEADR			; hold address message
	BANK0
	movwf	TabPointer

;------------- 	
; Message loop
Encoder_EEPROMinp_Loop1
	btfss	UCON,MSGF		; is sending enabled?
	return	; no: abort sending >>>>>>>>
	movfw	TabPointer
	BANK1
	movwf	EEADR
	BANK0

;-------------- 	
; One code loop
Encoder_EEPROMinp_Loop2
	btfss	UCON,MSGF		; is sending enabled?
	return					; no: abort sending >>>>>>>>

	call	ReadEENextChar	; move code into WREG and set STATUS bit Z
	btfsc	STATUS, Z		; Check if end of message #: if Z=0: no
	goto	Encoder_EEPROMinp_EOM; yes: end ====>
	
	xorlw	CS_ALIAS		; Check if call sign (special bit format)
	btfsc	STATUS, Z		; Z=1 if = CS	 
	goto	Encoder_EEPROMinp_CS; yes, ====>
	xorlw	CS_ALIAS		; restore code 
	call	Encoder			; Encode
	goto	Encoder_EEPROMinp_Loop2

	; Enf Of Message
Encoder_EEPROMinp_EOM
	movlw	0xFF			; infinity loop?
	xorwf	BeaconTime,W
	btfsc	STATUS,Z
	goto 	Encoder_EEPROMinp_Loop1	; for ever loop message======>

	decfsz	BeaconTime,f	; Loop beacon time ===>
	goto	Encoder_EEPROMinp_Loop1

	return

;-------------------------------------------- 	
; Extend CS and insert CALL SIGN into message
Encoder_EEPROMinp_CS 	
	BANK1
	movfw 	EEADR		; Save current TabPointer
	BANK0
 	; New current pointer
	movwf	TabPointerSave
	movlw	LOW(EE_CALLSIGN) - 1 
	BANK1
	movwf 	EEADR		
	BANK0
Encoder_CallSign_CS_Loop
	btfss	UCON,MSGF		; is sending enabled?
	return					; no: abort sending >>>>>>>>
	call	ReadEENextChar	; move code into WREG and set STATUS bit Z
	btfsc	STATUS, Z		; Check if end of call sign: if Z=0: no
	goto	Encoder_CallSign_CS_EOM	; yes: end ====>
	;call	getBitFrame		;
	call	Encoder			; Encode
	goto	Encoder_CallSign_CS_Loop
Encoder_CallSign_CS_EOM
	movfw 	TabPointerSave	; Restore previous pointer TabPointer
	BANK1
	movwf	EEADR
	BANK0
	goto	Encoder_EEPROMinp_Loop2

;---------------------- 	
Encoder_EEPROMinp_Error
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, ERROR_HM_NOT_FOUND	; send service message
	return

;---------------------------------------------------------------------- 	
;Encoder: Input from EEPROM: get header of message 
; Input: MsgN = message #
; Output: WREG = 0 if found or -1 if not found
; Output: EEADR = address of last character of header (BeaconTime)
Encoder_EEPROMinp_GetHead
	; Message loop <====
	call	ReadEENextChar
	xorlw	_HOM				; Check if header message
	btfss	STATUS, Z		; Z=0: no
	retlw	-1				; error ====>
	call	ReadEENextChar
	xorwf	MsgN,W			; Check if message #
	btfsc	STATUS, Z		; Z=0: no
	goto	Encoder_EEPROMinp_GetHead_Found; yes: found ====>
	; Code of Message loop <====
	call	ReadEENextChar	; Loop until End of message
	btfss	STATUS, Z		; At end Z=1
	goto 	$-2				; no: next code loop ====>
	;
	goto 	Encoder_EEPROMinp_GetHead; next message loop ====>
Encoder_EEPROMinp_GetHead_Found
	call	ReadEENextChar
	movwf	BeaconTime		; Hold beacon time
	retlw	0

;---------------------------------------------------------------------- 	
;Encoder: Input from RAM table 
; Input: WREG = start address character
Encoder_RAMinp
	movwf	TabPointer		; Holds text address
Encoder_RAMinp_Loop

	call	SERVICE_MSG_TABLE
	andlw	0xFF				; Check if at end of text
	btfsc	STATUS, Z			; zero returned at end
	return						; ===>if 0 end of text            

	call	Encoder				; Send character
	incf	TabPointer,f		; Point to next character
	goto	Encoder_RAMinp_Loop

;	endEncoder_RAMinp

;---------------------------------------------------------------------- 	
;Encoder 
; WREG contains Morse code to encode	
Encoder 	
	btfss	GPIO,GP_DIT
	goto	Encoder_Stop	; RB0 low = stop

	call	getBitFrame		; Transcode character to BitFrame
  	movwf   MorseCode       ; hold Morse code
  	movwf   ControlTape  	; hold Bit frame
	;
	; switch case Frame key
	;
	; case: 5 BITS (as number)
	andlw	b'00000111'		; strip off b0...b3
	xorlw	b'00000010'		; Check if 5 bit format
	btfsc	STATUS, Z		; Z=1 if = 5 bit format	 
	goto	Encoder_5b		; yes, ====>
	;
	; case: 6 BITS (as punctuation)
	movfw	ControlTape
	andlw	b'00000111'		; strip off b0...b3
	xorlw	b'00000011'		; Check if >6 bit special format
	btfsc	STATUS, Z		; Z=1 if = >6 bit special format	 
	goto	Encoder_Spe		; yes, ====>
	; 
	; case: 6 BITS SPECIAL
	movfw	ControlTape
	andlw	b'00000011'		; strip off b0...b2
	xorlw	b'00000001'		; Check if 6 bit format
	btfsc	STATUS, Z		; Z=1 if = 6 bit format	 
	goto	Encoder_6b		; yes, ====>
	; 
	; case: 4 BITS (letter)
	; 4 bit format, frame is in low nibble
  	movfw   ControlTape     ; 
	andlw	b'00001111'		; strip off low nibble
 	movwf   ControlTape  	; hold low nibble frame
	swapf	ControlTape,f	; low nibble nibble -> high nibble nibble
	; 

; Send generic morse code
;------------------------ 	
Encoder_loop 	
	rlf		ControlTape,f; carry -> bit 0, bit 7 -> carry
	btfss	STATUS,C		; if carry send DIT or DAH
	goto	Encoder_GAP		; EXIT>>>>>if no carry end of code
	; Send DIT or DAH
	rlf		MorseCode, f	; bit 7 -> carry
	btfss	STATUS,C		; if carry send DAH
	goto	Encoder_DIT
Encoder_DAH		
	call 	Player_DAH
	goto	Encoder_loop	; -->loop Morse code
Encoder_DIT	
	call 	Player_DIT
	goto	Encoder_loop 	; -->loop Morse code
Encoder_GAP	
	call 	Player_GAP
	return				  	; EXIT>>>>>

; Send 5 BITS morse code (as number)
;----------------------------------- 	
Encoder_5b
	movlw	b'11111000'		; Frame = 5 bits
	movwf	ControlTape 	
	goto	Encoder_loop  	; -->loop Morse code

; Send 6 BITS morse code (as puntuation)
;--------------------------------------- 	
Encoder_6b
	movlw	b'11111100'		; Frame = 6 bits
	movwf	ControlTape 	
	goto	Encoder_loop  	; -->loop Morse code

; Send 8 BITS morse code (as puntuation)
;--------------------------------------- 	
Encoder_8b
	movlw	b'11111111'		; Frame = 8 bits
	movwf	ControlTape 	
	bcf		STATUS,C		; To close 8 bits frame
	goto	Encoder_loop  	; -->loop Morse code

; Send 6 BITS SPECIAL morse code
;------------------------------- 	
Encoder_Spe
	;
	; switch case Frame
	;
	; case: WORD GAP
	movfw	ControlTape
	xorlw	_WG				; Check if word gap (special bit format)
	btfsc	STATUS, Z		; Z=1 if = word gap	 
	goto	Encoder_WGap	; yes, ====>
	;
	; case: AR
	movfw	ControlTape
	xorlw	_AR				; Check if AR (special bit format)
	btfsc	STATUS, Z		; Z=1 if = 	 
	goto	Encoder_AR		; yes, ====>
	;
	; case: BT
	movfw	ControlTape
	xorlw	_BT				; Check if BT (special bit format)
	btfsc	STATUS, Z		; Z=1 if = 	 
	goto	Encoder_BT		; yes, ====>
	;
	; case: SK
	movfw	ControlTape
	xorlw	_SK				; Check if SK (special bit format)
	btfsc	STATUS, Z		; Z=1 if = 	 
	goto	Encoder_SK		; yes, ====>
	;
	; case: ERROR
	movfw	ControlTape
	xorlw	_ERR			; Check if error (special bit format)
	btfsc	STATUS, Z		; Z=1 if = ERROR	 
	goto	Encoder_Error	; yes, ====>
	;
	; case: TUNEC
	movfw	ControlTape
	xorlw	_TUNEC			; Check if tune c (special bit format)
	btfsc	STATUS, Z		; Z=1 if = TUNEC	 
	goto	Encoder_TuneC	; yes, ====>
	;
	; case: TUNEP
	movfw	ControlTape
	xorlw	_TUNEP			; Check if tune c (special bit format)
	btfsc	STATUS, Z		; Z=1 if = TUNEP	 
	goto	Encoder_TuneP	; yes, ====>
	
; UNDEFINED CODE
;--------------- 	
	return					; EXIT>>>>>>>>>>>>undefined code

; Send WORD GAP
;-------------- 	
Encoder_WGap
	; 1+2+2+2=7 between 1 word (1 wait was played below sign yet+2 below code)
    bcf    	PORT_BUZ,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX	    	; Keyer Output To Low
  	movlw   WG - (1+2+2)
	movwf	WGapCount
Encoder_WGap_loop
	call   	TimerUnitP		; Wait for one unitperiod long
 	decfsz  WGapCount,f      
   	goto   Encoder_WGap_loop 	
	return	; EXIT>>>>>>>>>>>>>>>>6 BITS SPECIAL CODE

; Tune mode constant carrier
;--------------------------- 	
Encoder_TuneC
	call	Player_CARRIER	
	return					; no: abort sending
							; EXIT>>>>>>>>>>>>>>>>6 BITS SPECIAL CODE

; Tune mode 50% duty cycle dits
; Pulse can be speeded up or slowed down by button before start tune mode
;------------------------------ 	
Encoder_TuneP
	call	Player_DIT	
	return					; no: abort sending
							; EXIT>>>>>>>>>>>>>>>>6 BITS SPECIAL CODE
 
; _AR
;---- 	
Encoder_AR
	movlw	_AR5b			; Subtitute bit Frame for 5 bits format
	movwf	MorseCode
	goto	Encoder_5b
 
; _BT
;---- 	
Encoder_BT
	movlw	_BT5b			; Subtitute bit Frame for 5 bits format
	movwf	MorseCode
	goto	Encoder_5b
 
; _SK
;---- 	
Encoder_SK
	movlw	_SK6b			; Subtitute bit Frame for 6 bits format
	movwf	MorseCode
	goto	Encoder_6b
 
; Error 8 dots
;------------- 	
Encoder_Error
	movlw	_ERR8b	; Subtitute bit Frame for 8 bits format
	movwf	MorseCode
	goto	Encoder_8b
 
; Error 8 dots
;------------- 	
Encoder_Stop
   	bcf     UCON, MSGF		; unset flag for abort message sending
	goto	Encoder_Error

;	endEncoder

;------------------------------------------------------------ 	
; CW KEYER: Player 
;------------------------------------------------------------ 	
; Play one DOT or DAH related to TXF and BUZF flags
;------------------------------------------------------------ 	
; 					! TXF 				! NOT TXF			!
;------------------------------------------------------------ 	
; 					! BUZF 	! NOT BUZF	! BUZF 	 ! NOT BUZF	!
;------------------------------------------------------------ 	
; SEND TO TX		! Yes	! Yes		! No	 ! No		!
; SEND TO BUZZER 	! Yes	! No (1)	! Yes (2)! Yes (2)	!
;------------------------------------------------------------ 
; (1) Buzzer side tone off	
; (2) Only for service message	
;------------------------------------------------------------ 
Player_DIT	
	btfsc	UCON,TXF		; Tx Output?
	bsf		GPIO,GP_TX    
	btfsc	UCON,BUZF		; Buzzer Output?
	bsf    	PORT_BUZ,GP_BUZ			
    call   	TimerUnitP		; Wait for one Unit Period long
    bcf    	PORT_BUZ,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX	    	; Keyer Output To Low
	;movlw	(0xFF)^((1<<GP_BUZ)|(1<<GP_TX))
	;andwf	GPIO,f
    call   	TimerUnitP		; Wait for one Unit Period long
	return
Player_DAH		
	btfsc	UCON,TXF		; Tx Output To High?
	bsf		GPIO,GP_TX    
	btfsc	UCON,BUZF		; Buzzer Output To High?
	bsf    	PORT_BUZ,GP_BUZ			
    call   	TimerUnitP		; Wait for one Unit Period long
    call   	TimerUnitP		; Wait for one Unit Period long
    call   	TimerUnitP		; Wait for one Unit Period long
    bcf    	PORT_BUZ,GP_BUZ			; Keyer Output To Low
    bcf    	GPIO,GP_TX			; Keyer Output To Low
    call   	TimerUnitP		; Wait for one Unit Period long
	return
Player_GAP		
    ; 1+2 UnitPeriod between 2 codes (one Unit Period below one sign)
	call   	TimerUnitP		; Wait for one Unit Period long
    call   	TimerUnitP		; Wait for one Unit Period long
	return
Player_CARRIER ; special for tune mode		
	btfsc	UCON,TXF		; Tx Output?
	bsf		GPIO,GP_TX    
	btfsc	UCON,BUZF		; Buzzer Output?
	bsf    	PORT_BUZ,GP_BUZ			
	return

; End of Player
;---------------------------------------------------------------------- 	
;---------------------------------------------------------------------- 	
;---------------------------------------------------------------------- 	

;-------------------------------------------------------- 	
; 16 bit Compare A?B
; Input: CmpAH HIGH byte value A, CmpAL LOW byte value A, 
;		 CmpBH HIGH byte value B, CmpBL LOW byte value B
;		 Value register unchanged by routine
; Output: A_GT_B bit: 1 if A > B else 0
; Output: A_EQ_B bit: 1 if A = B else 0
; Output: A_LT_B bit: 1 if A < B else 0
;---------------------------------------------
; HIGH A > HIGH B 					!  A > B !
; HIGH A < HIGH B 					!  A < B !
; HIGH A = HIGH B AND LOW A > LOW B !  A > B !
; HIGH A = HIGH B AND LOW A = LOW B !  A = B !
; HIGH A = HIGH B AND LOW A < LOW B !  A < B !
;---------------------------------------------
CMP_A_B
	bsf		UCON1,A_GT_B		; init a > b
	bcf		UCON1,A_EQ_B		; init a not = b
	bcf		UCON1,A_LT_B		; init a not < b
	;
	; HIGH A > HIGH B ?
	movf	CmpAH,W
	subwf	CmpBH,W 	; W=CmpBH - CmpAH C=0 si CmpAH>CmpBH
    btfss   STATUS,C 	; CmpAH<=CmpBH: C=1
	return				; EXIT>>>>> A > B
	;
	; HIGH A = HIGH B ?
   	btfss   STATUS,Z 	
	goto	CMP_A_B_LT ; A < B
	;
	; CASE: HIGH A = HIGH B
	; 	lOW A > LOW B ?
	movf	CmpAL,W
	subwf	CmpBL,W 	; W=CmpBL - CmpAL C=0 si CmpAL>CmpBL
    btfss   STATUS,C 	; CmpAL<=CmpBL: C=1
	return				; EXIT>>>>>  A > B
	;
	; CASE: HIGH A = HIGH B
	; 	lOW A = LOW B ?
  	btfss   STATUS,Z 	
	goto	CMP_A_B_LT
	;
	; CASE: HIGH A = HIGH B AND	lOW A = LOW B
	bsf		UCON1,A_EQ_B		
 	bcf		UCON1,A_GT_B
	return				; EXIT>>>>> A = B
	;
	; CASE: A < B
CMP_A_B_LT
 	bcf		UCON1,A_GT_B
 	bsf		UCON1,A_LT_B
	return	; EXIT>>>>> a<b

;---------------------------------------------------------------------- 	
; KPAD_GetADCvalue
; FROM MICROCHIP AN700 Filename: DeltaSig.asm 	
;---------------------------------------------------------------------- 	
; Convert analog voltage level to 10 bits number
; Input: RA0 RA3
; Output: 10 bit ADC value ADRESL ADRESH
; mininum conversion time is 21mS @4mHz.

KPAD_GetADCvalue
	clrf	counter
	clrf	counter+1
	clrf	ADRESL
	clrf	ADRESH
	; CM2:CM0 = Two Common Reference Comparators	
	movlw	0x03	; set up for 2 analog comparators with common reference	
	movwf	CMCON
loop
	btfsc	CMCON, C1OUT	; Is comparator high or low?	
	goto	complow			; Go the low route	

comphigh
	nop						; necessary to keep timing even	
	bcf	PORTA, 3			; 'DAC' = 0
	incfsz	ADRESL,f		; bump counter
	goto	eat2cycles
	incf	ADRESH, f
	goto	endloop

complow
	bsf	PORTA, 3			; 'DAC' = 1
	nop						; necessary to keep timing even	
	goto	eat2cycles
eat2cycles
	goto	endloop			; eat 2 more cycles	

endloop
	incfsz	counter,f		; Count this lap through the loop.	
	goto	eat5cycles
	incf	counter+1,f
	movf	counter+1,w
	andlw	0x04		; Are we done 1024 ? (We're done when bit2 of	
	btfsc	STATUS, Z	; the high order byte overflows to 1).	
	goto	loop		; no
	goto	exit		; yes
eat5cycles
	goto	$+1			; more wasted time to keep the loops even	
	nop
	goto	loop

exit
	movlw	0x06		; set up for 2 analog comparators with common reference	
	movwf	CMCON		; Comp1 out = RA3

    return

;------------------------------------- 	
; KPAD_Keystroke process
; Scan ADC KPAD channel until find key
; Execute command associated with key
KPAD_Keystroke
	clrf	INTCON				; disable all interrupt
    bcf  	UCON, KPADF 		; keypad locked

KPAD_GetADRES
	;movlw	ADC_CON | ADC_AN_KPAD	; select channel Keypad
	movlw	D'60'
	movwf	SampleCount
	call	KPAD_GetADCvalue
	call	KPAD_GetADCvalue
KPAD_GetADRES_10	
    movf	ADRESL,W       	; 8 bits low
	movwf	CmpBL
	movf	ADRESH,W        ; 2 bits High	
	movwf	CmpBH
	call	KPAD_GetADCvalue	
	;BANK1					; store ADC result into compare register A
    movf	ADRESL,W       	; 8 bits low
	;BANK0
	movwf	CmpAL
	movf	ADRESH,W        ; 2 bits High	
	movwf	CmpAH
 	call	CMP_A_B			; compare ADC A:B
	btfss	UCON1,A_GT_B	; is A>B?
	goto	KPAD_GetADRES_20; no
	decfsz	SampleCount,f
	goto	KPAD_GetADRES_10; yes

KPAD_GetADRES_20
	movfw	CmpBL
	movwf	ADRESL	
  	movwf	CmpAL
	movfw	CmpBH
	movwf	ADRESH	
	movwf	CmpAH
	; point to first ADC table item
	movlw	ADC_Vth_KEYPAD - ADC_Vth_TABLE - 1 -3 -3 	; W = value relative address - 3
	movwf	ADCVthPointer		; Hold pevious line address

KPAD_Lookup	; read one table item
	incf	ADCVthPointer,f		; skip min value of previous table item
	incf	ADCVthPointer,f			 
	incf	ADCVthPointer,f		; point to current table item
	call	ADC_Vth_TABLE		; lookup voltage scale to find key pressed
	movwf   KeyPadXYtemp       	; hold KeyPadXY byte

	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
 	movwf   CmpBH       		; hold MAX value high byte into compare register B
	
	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
   	movwf   CmpBL	       		; hold MAX value low byte into compare register B

 	call	CMP_A_B				; compare ADC : MAX value
	btfss	UCON1,A_LT_B		; is ADC value < MAX table value ?
	goto	KPAD_Lookup			; no: NEXT item------>

	; value is < MAX: then look for MIN value
	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
 	movwf   CmpBH       		; hold MIN value high byte into compare register B
	
	incf	ADCVthPointer,f		; Point to next byte
	call	ADC_Vth_TABLE
   	movwf   CmpBL	       		; hold MIN value low byte into compare register B

 	call	CMP_A_B				; compare ADC : MIN value
	btfsc	UCON1,A_EQ_B		; is ADC value = MIN table value
	goto	KPAD_GetKey_Found	; yes: Key found------>
	btfss	UCON1,A_GT_B		; is ADC value > MIN table value
	goto	KPAD_GetADRES		; no: value out of voltage span, read again------>

KPAD_GetKey_Found
 	movf	KeyPadXY,W			; Hold previous key
	movwf	KeyPadXYPre			
 	movf	KeyPadXYtemp,W		; Hold new key
	movwf	KeyPadXY			

;-------------------------------
; only for analyze and benchmark
KPAD_Report	
	ifdef __ADRES_TRON	
		; Report ADRES value into EEPROM for analyse
		addwf	KeyPadXY,W		; pointer = 2*KeyPadXY
		btfsc	STATUS,C		; carry set if key 0xFF
		goto	KPAD_Report_End	; >>>no report
		sublw	LOW(EE_ADRESL0); pointer to LSB
		BANK1
		movwf	EEADR			; into EEADR
		BANK0
		movf	ADRESL,W		; ADC value LSB into EEDATA
		BANK1
		movwf	EEDATA			
		call	WriteEEPROM		; write one byte to EEPROM
		BANK1
		decf	EEADR,f			; pointer to MSB
		BANK0
		movf	ADRESH,W		; ADC value MSB into EEDATA
		BANK1
		movwf	EEDATA			
		call	WriteEEPROM		; write one byte to EEPROM

	endif
KPAD_Report_End
;--------------

	; switch case KeyPadXY
	;
	; case: NO KEY
	movfw	KeyPadXY
	xorlw	KPAD_FF				; Check if no key
	btfsc	STATUS, Z			; Z=0: no next key	 
	goto	KPAD_Keystroke_End	; Z=1: yes, end key ====>
	;
	; case: KEY = PREVIOUS KEY
	movfw	KeyPadXY
	xorwf	KeyPadXYPre,W		; Check if equal
	btfsc	STATUS, Z			; Z=0: no
	goto	KPAD_Keystroke_End	; end key ====>
	;
	; case: SEND0
	movfw	KeyPadXY
	xorlw	KPAD_SEND0		; Check if key Send0
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_Sendn	; yes, ====>
	;
	; case: SEND1
	movfw	KeyPadXY
	xorlw	KPAD_SEND1		; Check if key Send1
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_Sendn	; yes, ====>
	;
	; case: SEND2
	movfw	KeyPadXY
	xorlw	KPAD_SEND2		; Check if key Send2
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_Sendn	; yes, ====>
	;
	; case: BUZZER
	movfw	KeyPadXY
	xorlw	KPAD_BUZ		; Check if key Buz
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_Buz	; yes, ====>
	;
	; case: TUNEP
	movfw	KeyPadXY
	xorlw	KPAD_TUNEP		; Check if key Tune PULSE
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_TunP	; yes, ====>
	;
	; case: TUNEC
	movfw	KeyPadXY
	xorlw	KPAD_TUNEC		; Check if key Tune CARRIER
	btfsc	STATUS, Z		; Z=0: no	 
	goto	KPAD_Keystroke_TunC	; yes, ====>
	;
	; case: MODEAB
	movfw	KeyPadXY
	xorlw	KPAD_MOD		; Check if key Mod
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_Mod	; yes, ====>
	;
	; case: SPEED UP
	movfw	KeyPadXY
	xorlw	KPAD_SPEED_UP	; Check if key Speed up
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_SpeedUp	; yes, ====>
	;
	; case: SLOW_DOWN
	movfw	KeyPadXY
	xorlw	KPAD_SLOW_DOWN	; Check if key Slow down
	btfsc	STATUS, Z		; Z=0: no
	goto	KPAD_Keystroke_SlowDown	; yes, ====>
	;
KPAD_Keystroke_End ; End of keypad
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD


; Send message #0,#1,#2 to Tx
;---------------------------- 	
KPAD_Keystroke_Sendn
	ifdef __TRON
		movlw	"S"
		call	Encoder
	endif

	btfss	UCON,BUZE		; If side tone buzzer disable
	bcf		UCON,BUZF		; unset flag buzzer
	bsf		UCON,TXF		; set flag Tx

	movfw	KeyPadXY ; Button assign = message #
	movwf	MsgN

	bsf		UCON,MSGF		; enable message sending
							; sending may be interrupted by DIT paddle
	;bcf		INTCON,INTF
	;bsf		INTCON,INTE		; enable interrupt by DIT paddle
	;bsf		INTCON,GIE
	; SEND ON
	call 	Encoder_EEPROMinp

	;bcf		INTCON,GIE
	;bcf		INTCON,INTE
	;bcf		INTCON,INTF
	bsf		UCON,BUZF		; set flag buzzer
	bcf		UCON,TXF		; unset flag Tx
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Side Tone Buzzer ON/OFF
; Only for TX message
; Not for service message
;------------------------ 	
KPAD_Keystroke_Buz
	movlw  	(1 << BUZE)
    xorwf  	UCON,f  ; Toggle buzzer bit
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, BUZ_MSG	; send service message
	call 	SaveOption
	return

; Tune mode constant carrier
;--------------------------- 	
KPAD_Keystroke_TunC
	btfss	UCON,BUZE		; If side tone buzzer disable
	bcf		UCON,BUZF		; unset flag buzzer
	bsf		UCON,TXF		; set flag Tx

	bsf		UCON,MSGF		; enable message sending
							; sending may be interrupted by DIT paddle
	;bcf		INTCON,INTF
	;bsf		INTCON,INTE		; enable interrupt by DIT paddle
	; carrier ON
	movlw	_TUNEC
	call	Encoder
	
	btfsc	GPIO,GP_DIT		; DIT high/low = is sending enabled? <----
	goto	$-1				; yes: loop -------------->
	; carrier OFF
    bcf    	PORT_BUZ,GP_BUZ		; Keyer Output To Low
    bcf    	GPIO,GP_TX			; Keyer Output To Low

	;bcf		INTCON,INTE		; disable interrupt DIT paddle

	bcf		UCON,TXF		; unset flag Tx
	bsf		UCON,BUZF
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Tune mode pulse 50% duty cycle dits
;------------------------------------ 	
KPAD_Keystroke_TunP
	ifdef __TRON
		movlw	"P"
		call	Encoder
	endif

	btfss	UCON,BUZE		; If side tone buzzer disable
	bcf		UCON,BUZF		; unset flag buzzer
	bsf		UCON,TXF		; set flag Tx

	bsf		UCON,MSGF		; enable message sending
							; sending may be interrupted by DIT paddle
	;bcf		INTCON,INTF
	;bsf		INTCON,INTE		; enable interrupt by DIT paddle
	; Pulse ON
	movlw	_TUNEP			; <-------------------------
	call	Encoder
	btfsc	UCON,MSGF		; is sending enabled?
	goto	$-3				; yes: loop  -------------->
	; Pulse OFF

	;bcf		INTCON,INTE		; disable interrupt DIT paddle

	bcf		UCON,TXF		; unset flag Tx
	bsf		UCON,BUZF
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Toggle mode A or B
;------------------- 	
KPAD_Keystroke_Mod
	movlw  	(1 << MOD_B)
    xorwf  	UCON,f        	; Toggle mod bit
	call 	SaveOption
	btfss	UCON,MOD_B
	goto	KPAD_Keystroke_Mod_A
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, MOD_B_MSG	; send service message
	return
KPAD_Keystroke_Mod_A
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, MOD_A_MSG	; send service message
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Speed up
;--------- 	
KPAD_Keystroke_SpeedUp
	movf	WordPerMn,W		; Check if max value
	xorlw	SPEED_MAX			
	btfsc	STATUS, Z		; Z=1 returned at end
	return	; EXIT>>>>>end of speed scale table            

	incf	WordPerMn,f
	movfw	WordPerMn
	call	getUnitPeriod		; get UnitPeriod 
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, SPEED_UP_MSG	; send service message
	call 	SaveOption
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

; Slow down
;---------- 	
KPAD_Keystroke_SlowDown
	movf	WordPerMn,W		; Check if min value
	xorlw	SPEED_MIN			
	btfsc	STATUS, Z		; Z=1 returned at end
	return	; EXIT>>>>>end of speed scale table            

	decf	WordPerMn,f
	movfw	WordPerMn
	call	getUnitPeriod	; get UnitPeriod 
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, SLOW_DOWN_MSG	; send message
	call 	SaveOption
	return	;>>>>>>>>>>>>>>>>END OF KEYPAD

;---------------------------------------------------------------------- 	
; Automatic set type of key: iambic or straight
; Input: GPIO,GP_DAH	
; Output: UCON,IAMBIC_KEY flag	
; iambic key: if DAH not grounded
; straight key if DAH grounded
SetKeyType
	btfss	GPIO,GP_DAH				; iambic or straight key?
	goto	SetKeyType_S
	;
	bsf		UCON,IAMBIC_KEY 	
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, IAMBIC_MSG	; iambic message
	return	; >>>>>>>>>>>>
SetKeyType_S
	bcf		UCON,IAMBIC_KEY 	
	SEND_SERVICE_MSG SERVICE_MSG_TABLE, STRAIGHT_MSG; straight message
	return

;---------------------------------------------------------------------- 	
; Set TIMER1 interrupt
; TIMER1 is settle for 1s wait
SetIntTIMER1
    
 	BANK0					
    btfsc  	UCON, KPADF 	; Is keypad locked?
	return					; no: no more interrupt>>>>>>

	movlw  	TMR1_OVF_INIT  		; set max TMR1 overflow
	movwf	TMR1ovf
    movlw   LOW(TMR1_INIT)		; clear TRM1 refore reset flag
    movwf   TMR1L
    movlw   HIGH(TMR1_INIT)
    movwf   TMR1H
	
	bcf     PIR1, TMR1IF    	; clear TMR1 interrupt flag
    banksel PIE1
	bsf     PIE1, TMR1IE   		; enable TMR1 interrupts
	BANK0
	bsf		INTCON,PEIE
	bsf		INTCON,GIE
	return

;------------------------- 	
; Load options from EEPROM									
;
LoadOption
	; Set last value of WPM
	movlw	LOW(EE_WPM)
	call	ReadEEChar
	movwf	WordPerMn		; into WordPerMn option
	
	; Set last value of User Program control Register
	movlw	LOW(EE_UCON)	
	call	ReadEEChar
	movwf	UCON			; into Program Control Register

	; Set 1st message start address
	; 1st message is next CALLSIGN
	movlw	LOW(EE_BUFFER)-1
	BANK1
	movwf	EEADR	

	;read until WREG=0
	incf	EEADR,f
	bsf		EECON1,RD		;read
	movf	EEDATA,W		;data ouput Z=1 if zero
	btfss	STATUS,Z
	goto	$-4	;------>
	;end of CALLSIGN
	movfw	EEADR
	BANK0
	movwf	EEMessagePtr
	incf	EEMessagePtr,f	;Hold Ptr

	return

;---------------------------------------------------------------------- 	
; EEPROM GENERAL SUBROUTINE
;---------------------------------------------------------------------- 	
;----------------------------------------- 	
; Write EEPROM	
; Input EEDAT: data to put in EEPROM
; Input	EEADR: address of data into EEPROM
;
WriteEEPROM
	BANK1
	bcf EECON1,EEIF 	;disable interrupt write to EEPROM
	bsf EECON1,WREN 	;enable write to EEPROM
	movlw 0x55
	movwf EECON2
	movlw 0xaa
	movwf EECON2
	bsf EECON1,WR		; set WR = write EEPROM
WriteEEwait
	btfsc EECON1,WR		; loop <---------------
	goto $-1			; loop until WR = 0 -->
	bcf EECON1,WREN 	; disable write to EEPROM
	BANK0
	return 

;---------------------------------------------------------------------- 	
; Save options into EEPROM	
;
SaveOption
	movf	WordPerMn,W			; WPM into EEDATA
	BANK1
	movwf	EEDATA			
	movlw	LOW(EE_WPM)			; EEPROM WPM output address
	movwf	EEADR
	call 	WriteEEPROM

	BANK0
	movf	UCON,W				; User Flag into EEDATA
	BANK1
	movwf	EEDATA			
	movlw	LOW(EE_UCON)		; EEPROM User Flag output address
	movwf	EEADR
	call 	WriteEEPROM
	BANK0
	return 

;---------------------------------------------------------------------- 	
; Read character from EEPROM @address 
; Input: WREG = relative address
; Output: WREG= character
ReadEEChar
	BANK1
	movwf	EEADR	
	bsf		EECON1,RD		; read
	movf	EEDATA,W		; data ouput Z=1 if zero
	BANK0
	return

;---------------------------------------------------------------------- 	
; Read next character from EEPROM
; Input: EEADR = address of previous character
; Output: WREG = character @EEADR+1
; Output: EEADR + 1= address of current character
ReadEENextChar
	BANK1
	movlw	EEPROM_DATA_MAX		; Check if out of memory
	xorwf	EEADR,W				; Check if out of memory
	btfsc	STATUS, Z			; Z=1 returned if error not found
	goto	ReadEENextChar_Error
	incf	EEADR,f
	bsf		EECON1,RD		; read
	movf	EEDATA,W		; data ouput Z=1 if zero
	BANK0
	return
ReadEENextChar_Error ; Out memory protect
	BANK0
	retlw	_ERR ; return with STATUS Z bit below = 1
	return

;---------------------------------------------------------------------- 	
; FROM MICROCHIP AN700 Filename: DeltaSig.asm 	
;---------------------------------------------------------------------- 	
;* Author:         Dan Butler 	
;* Company:        Microchip Technology Inc. 	
;* Revision:       1.00 	
;* Date:           02 December 1998 	
;* Assembled using MPASM V2.20 	
;********************************************************************* 	
;* Include Files: 	
;*         p16C622.inc     V1.01 	
;********************************************************************* 	
;* Provides two functions implementing the Delta Sigma A2D. 	
;* InitDeltaSigA2D sets up the voltage reference and comparator 	
;* in the "idle" state. 	
;* DeltaSigA2D runs the actual conversion.  Results provided in 	
;* result_l and result_h. 	
;* See An700 figure 2 for external circuitry required. 	
;---------------------------------------------------------------------- 	

InitDeltaSigA2D	
	; Settle Voltage Reference register
	banksel	VRCON	;0xE6
	movlw	B'11101100'
    ;         1-------  VREN: VREF Enable 
	;						1 = VREF circuit powered on 	
	;						0 = VREF circuit powered down, no IDD drain 	
    ;         -1------  VROE: VREF Output Enable
 	;						1 = VREF is output on RA2 pin 	
	;						0 = VREF is disconnected from RA2 pin 	
   	;         --1-----  VRR: VREF Range selection
	;						1 = Low Range 	
	;						0 = High Range 	
    ;         ---0---- 	Unimplemented: Read as '0'
    ;         ----1100  VR<3:0>: VREF value selection 0 = VR [3:0] = 15
	;					When VRR = 1: VREF = (VR<3:0>/ 24) * VDD 	
	;					When VRR = 0: VREF = 1/4 * VDD + (VR<3:0>/ 32) * VDD	
	movwf	VRCON		

	bcf		TRISA,RA3	;set comparator pin to output	

	;set up for 2 analog comparators with common reference	
	banksel	CMCON	;0x06
	movlw	B'00000110'
    ;         0-------  C2OUT: Comparator 2 Output
	;						When C2INV = 0: 	
	;						1 = C2 VIN+ > C2 VIN- 	
	;						0 = C2 VIN+ < C2 VIN- 	
	;						When C2INV = 1: 	
	;						1 = C2 VIN+ < C2 VIN- 	
	;						0 = C2 VIN+ > C2 VIN- 	
    ;         -0------  C1OUT: Comparator 1 Output
	;						When C1INV = 0: 	
	;						1 = C1 VIN+ > C1 VIN- 	
	;						0 = C1 VIN+ < C1 VIN- 	
	;						When C1INV = 1: 	
	;						1 = C1 VIN+ < C1 VIN- 	
	;						0 = C1 VIN+ > C1 VIN- 	
   	;         --0-----  C2INV: Comparator 2 Output Inversion 
	;						1 = C2 Output inverted  	
	;						0 = C2 Output not inverted 	
   	;         ---0----  C1INV: Comparator 1 Output Inversion
	;						1 = C1 Output inverted  	
	;						0 = C1 Output not inverted 	
    ;         ----0--- 	CIS: Comparator Input Switch'
	;						When CM2:CM0: = 001 	
	;						Then: 	
	;						1 = C1 VIN- connects to RA3 	
	;						0 = C1 VIN- connects to RA0 	
	;						When CM2:CM0 = 010 	
	;						Then: 	
	;						1 = C1 VIN- connects to RA3 	
	;      						C2 VIN- connects to RA2 	
	;						0 = C1 VIN- connects to RA0 	
	;      						C2 VIN- connects to RA1 	
    ;         -----110  CM2:CM0: Comparator Mode
	;					Figure 9-1 shows the Comparator modes and CM2:CM0 bit settings	
	;					Two Common Reference Comparators with Outputs	
	movwf	CMCON	
	return

;------------
UserProgStart
;------------
Main
	; Set up Oscillator calibration value
	;call    OSCCAL_FACTORY_ADR	; retrieve factory calibration value
	;banksel OSCCAL				; and save onto register
	;movwf   OSCCAL      

    ;banksel GPIO
    ;movlw   b'00000000'	; all ports low level
    ;movwf   GPIO

	; set up I/O port direction
    banksel TRISB			;1111 1111 Power on
	bcf		TRISB,GP_TX		;TX control
    banksel TRISB			;1111 1111 Power on
	bcf		TRISB,GP_BUZ	;Buzzer control

	banksel	PORTB			;TX output and Buzzer ouput Low
	bcf		PORTB,GP_TX
	banksel	PORT_BUZ		;TX output and Buzzer ouput Low
	bcf		PORT_BUZ,GP_BUZ

	; set up option register
	banksel OPTION_REG		;1111 1111 Power On
    movlw   b'11111111'
    ;         1-------  NOT GPPU: 1 = GPIO pull-ups are disabled
    ;         -1------  INTEDG: 1 = Interrupt on rising edge of GP2/INT pin
    ;         --1-----  T0CS: 1 = Transition on GP2/T0CKI pin0=output, GP4, Buzzer control
    ;         ---1----  T0SE: 1 = Increment on high-to-low transition on GP2/T0CKI pinnot used, GP3, MCLR and reset
    ;         ----1---  PSA: 1 = Prescaler is assigned to the WDT1=input, GP4, DAH paddle input
    ;         -----111  PS2:PS0: 111= 1:128 prescaler
  	movwf	OPTION_REG	

	; set up comparator
	; set up A/D converter
	call	InitDeltaSigA2D	

	; initialize interrupts
    banksel INTCON
    movlw   b'01000000'
    ;         0-------  0=interrupts disabled
    ;         -0------  0/1=enable/disable peripheral interrupts
    ;         --0-----  0=disable TMR0 overflow interrupt
    ;         ---0----  0=disable GP2/INT external interrupt
    ;         ----0---  0=disable GPIO port change interrupt
    ;         -----0--  0=no on TMR0 overflow
    ;         ------0-  0=no GP2/INT external interrupt
    ;         -------0  0=no GPIO port change
    movwf   INTCON
    
    banksel PIE1 	; in bank 1   
    movlw   b'00000000'
    ;         0-------  0=disable EE write complete interrupt
    ;         -0------  0=disable A/D converter interrupt
    ;         --xx----  not implemented
    ;         ----0---  0=comparator interrupt disabled
    ;         -----xx-  not implemented
    ;         -------0  0=disable TMR1 overflow interrupt
    movwf   PIE1
        
    banksel PIR1	; in bank 0
    movlw   b'00000000'
    ;         0------- 0=no EE write complete
    ;         -0------ 0=no A/D conversion complete
    ;         --xx---- not implemented
    ;         ----0--- 0=no comparator interrupt
    ;         -----xx- not implemented
    ;         -------0 0=no TMR1 overflow
    movwf   PIR1

	; initialize TMR1
    banksel T1CON
    movlw  b'00110101'    
    ;        x-------  not implemented
    ;        -0------  0=gate disabled
    ;        --11----  01=1:8 prescaler
    ;        ----0---  0=LP oscillator is off
    ;        -----1--  1=external clock input not synchronized
    ;        ------0-  0=internal clock
    ;        -------1  1=enable timer
    movwf   T1CON
     
	BANK0

	clrf	UCON1				; All user flags 0
	movlw	KPAD_FF				; Key pad = no key pushed
	movwf	KeyPadXY
	movwf	KeyPadXYPre

	call	LoadOption			; Load options from EEPROM
	
	movfw	WordPerMn			
	call	getUnitPeriod		; get UnitPeriod

	BANK0

	clrf	TMR1L			; TMR1=0
   	clrf	TMR1H
	;bsf     INTCON,GIE     	; enable all interrupts

	call	SetKeyType		; Iambic or Straight key

;----------------------------------------------------------
; TMR1IE overflow interrupts are disabled by any DIT or DAH 
; Keypad is locked by any DIT or DAH
Main_Loop
	btfss	UCON,IAMBIC_KEY	; 2 WAYS PROCESS: IAMBIC OR STRAIGHT KEY
	goto 	StraightKey

;----------------------------
; IAMBIC KEY MODE A OR MODE B
;----------------------------
IambicKey
	BANK0
	movfw	GPIO							; GPIO snapshot
	andlw	(1 << GP_DIT)|(1 << GP_DAH) 	; strip off paddle port
	movwf	GPIOcopy
	;
	; switch case GPIOcopy
	;
	; case: DIT KEY DOWN
	movfw	GPIOcopy
	xorlw	(0 << GP_DIT)|(1 << GP_DAH)	; Check if DIT down
	btfsc	STATUS, Z		; Z=0: no
	goto	IambicKey_DIT	; end key ---->
	;
	; case: DAH KEY DOWN
	movfw	GPIOcopy
	xorlw	(1 << GP_DIT)|(0 << GP_DAH)	; Check if DAH down
	btfsc	STATUS, Z		; Z=0: no
	goto	IambicKey_DAH	; end key ---->
	;
	; case: DIT and DAH SQUEEZE BOTH
	movfw	GPIOcopy
	xorlw	(0 << GP_DIT)|(0 << GP_DAH)	; Check if 2 keys down
	btfsc	STATUS, Z			; Z=0: no next key	 
	goto	IambicKey_BOTHPAD	; Z=1: yes, end key ---->
	;
	; case: NO DIT and NO DAH
	btfss	UCON1,BOTHPADF
	goto	KPAD_Interface				; no key down: scan keypad?=======>
	btfss	UCON,MOD_B
	goto	KPAD_Interface				; no key down: scan keypad?=======>
IambicKey_RELEASE
 	; Mode B keyer sends an additional element opposite 	
	; to the one being sent when the paddles are released.	
	movlw  	(1 << IAMBIC_SW)
    xorwf  	UCON1,f        	; Toggle iambic bit
	btfss	UCON1,IAMBIC_SW
	goto	IambicKey_DAH
IambicKey_DIT
	; yes! send one DIT	
	bcf 	UCON1,BOTHPADF 	; Both paddles released
IambicKey_DIT_Both
	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupt
	BANK0
	bsf		UCON1,IAMBIC_SW		; settle Iambic switch
	bcf		UCON, KPADF 		; Lock Keypad
	bsf		UCON,TXF			; set flag Tx
	call 	Player_DIT
	bcf		UCON,TXF			; unset flag Tx
	goto	IambicKey  			; loop for another input ---->
IambicKey_DAH
	bcf 	UCON1,BOTHPADF 		; Both paddles released
IambicKey_DAH_Both
 	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupt
	BANK0
	bcf		UCON1,IAMBIC_SW		; settle Iambic switch
    bcf		UCON, KPADF 		; Lock Keypad
	bsf		UCON,TXF			; set flag Tx
	call 	Player_DAH
	bcf		UCON,TXF			; unset flag Tx
	goto	IambicKey  			; Loop for another input ---->
IambicKey_BOTHPAD
	movlw  	(1 << IAMBIC_SW)
    xorwf  	UCON1,f        		; Toggle iambic bit
	bsf 	UCON1,BOTHPADF 		; Both paddles squeezed
	btfss	UCON1,IAMBIC_SW
	goto	IambicKey_DAH_Both	; ---->
	goto	IambicKey_DIT_Both  ; ---->

;-------------
; STRAIGHT KEY
;-------------
StraightKey
	btfsc	GPIO,GP_DIT
	goto	StraightKey_Up		; no key down: ---->
;	Straight key down
	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupt
	BANK0
    bcf		UCON, KPADF 		; Lock Keypad
	bsf    	GPIO,GP_TX	   		; Tx Output
	btfsc	UCON,BUZE			; Buzzer Output Enable?
	bsf    	PORT_BUZ,GP_BUZ			
	goto	StraightKey
StraightKey_Up
	bcf    	GPIO,GP_TX	    	; Tx Output
	bcf    	PORT_BUZ,GP_BUZ	   	; Buzzer Output
	goto	KPAD_Interface		; no key down: scan keypad? ====>

;-------------------------------- 	
; SCAN KEYPAD ON INTERRUPT TIMER1
;-------------------------------- 	
KPAD_Interface
    btfss  	UCON, KPADF 		; Is keypad unlocked?
	goto 	KPAD_Interface_TMR1	; no: set TIMER1 ------>
	
	;call	KPAD_GetKey			; yes! keypad ready to use
	call 	KPAD_Keystroke
	goto	Main_Loop			; Loop for another input ------>

KPAD_Interface_TMR1
	banksel PIE1
	btfss	PIE1, TMR1IE   		; Is TMR1 interrupt ready?
	call 	SetIntTIMER1		; no: get it ready
	BANK0						; yes: wait for interrupt
	goto	Main_Loop			; Loop for another input ------>

;------------------ 	
; INTERRUPT MANAGER
;------------------ 	
ISR
	;save register
	movwf	ISRwreg  		; WREG
	swapf	STATUS,w		; STATUS without change Z
	movwf	ISRstatus		; 
        
ISR_TMR1
 	; check for TMR1 overflow interrup switch
	; beware of bank! PIR1 is in bank0, PIE1 in bank1
	banksel PIE1
	btfss	PIE1,TMR1IE			; interrupt timer enable ?
	goto 	ISR_TMR1_END		; no: next interrupt

	; yes: interrupt timer running?
    BANK0
	btfss	PIR1,TMR1IF		
	goto 	ISR_TMR1_END		; no: end of interrupt

	; yes TMR1 overflow ?
    decfsz  TMR1ovf,f
	goto	ISR_TMR1_rollover	; no: counter rollover

    ; end of count
	banksel PIE1
    bcf     PIE1, TMR1IE    	; disable overflow interrupts
    BANK0
	bcf     PIR1, TMR1IF    	; clear interrupt flag
    bsf     UCON, KPADF  		; unlock keypad to main loop control
	goto	ISR_TMR1_END

ISR_TMR1_rollover
    movlw   LOW(TMR1_INIT)		; reset TMR1
    movwf   TMR1L
    movlw   HIGH(TMR1_INIT)
    movwf   TMR1H
    bcf     PIR1, TMR1IF    	; clear interrupt flag
ISR_TMR1_END

 	; check for GP2 interrup switch
	; -----------------------------
ISR_GP2
ISR_GP2_END
ISRnext
	; other interrupt
	goto	ISRend				; end of interrupt

;restore register
ISRend
	swapf	ISRstatus,w		; STATUS without change Z
	movwf   STATUS				
	swapf   ISRwreg,f		; WREG without change Z
 	swapf   ISRwreg,w  			
	retfie  				; return from interrupt set GIE
	
;---------------------------------------------------------------------- 	
	org		MSG_DATA
;---------------------------------------------------------------------- 	
; Message Table to send
; Beware of bloc page (PCLATH change)
; input: ADCVthPointer = Relative character address
; output: WREG
;---------------------------------------------------------------------- 	
SERVICE_MSG_TABLE
	movlw 	HIGH(MSG_DATA) ; PCLATH = HIGH bit address
	movwf 	PCLATH 			;		
	movf	TabPointer,W	; for PCL
	addwf	PCL ,F 			; Jump to character pointed to in W register
SERVICE_MSG
;
; SERVICE MESSAGE
;
IAMBIC_MSG		; iambic key
	DT "I",_EOM
STRAIGHT_MSG	; straight key
	DT "S",_EOM
SPEED_UP_MSG	; signal for speed change up
	DT "U",_EOM
SLOW_DOWN_MSG	; signal for speed change down
	DT "D",_EOM
MOD_A_MSG		; signal for mode A change
	DT "A",_EOM
MOD_B_MSG		; signal for mode B change
	DT "B",_EOM
BUZ_MSG			; signal for sidetone buzzer on/off
	DT _AR,_EOM
ERROR_HM_NOT_FOUND	; signal for error header message not found
	DT _ERR,_EOM
;
	IF ((HIGH ($)) != (HIGH (SERVICE_MSG)))
    	ERROR "SERVICE_MSG_TABLE CROSSES PAGE BOUNDARY!"
	ENDIF;

;---------------------------------------------------------------------- 	
	;org		OSCCAL_FACTORY_ADR
;---------------------------------------------------------------------- 	

;====================================================================== 	
;====================================================================== 	
; USER PROGRAM END
;====================================================================== 	
;====================================================================== 	


;---------------------------------------------------------------------- 	
; EEPROM LOADER ROOT 
;---------------------------------------------------------------------- 	
	ORG	ELoaderRoot

ELoaderExit			       
	clrf	PCLATH	       ; Don't remove set PCLATH on PAGE 0

	; Set PCLATH on user program start page and run
	;pagesel ELoaderExit	 
	;goto 	ELoaderExit
	pagesel UserProgStart
	goto 	UserProgStart ;>>>>>>>>>>>>>>>EXIT

;--------------
	ORG	ELoaderRoot+5

ELoaderStart
	btfss	STATUS,NOT_TO	; WatchDog-TimeOut ?
	goto	ELoaderExit

	;-------------
	; Settle USART
	;-------------
	; set up RCSTA: RECEIVE STATUS AND CONTROL REGISTER	
	BANKSEL	RCSTA
	movlw	B'10010000'
    ;         1-------  SPEN 1=Serial port enabled
    ;         -0------  RX9 0=Selects 8-bit reception
    ;         --0-----  SREN 0=don't care in asynchronous mode
    ;         ---1----  CREN 1=in Asynchronous mode, Enables continuous receive
    ;         ----0---  ADEN 0=in Asynchronous mode 9-bit (RX9 = 1):	
	;						Disables address detection, all bytes are received, 
	;						and ninth bit can be used as PARITY bit	
    ;         -----0--  FERR 0=No framing error
    ;         ------0-  OERR 0=No overrun error
    ;         -------0  RX9D 0=9th bit of received data. Can be PARITY bit	
	movwf	RCSTA

	; set up TXSTA: TRANSMIT STATUS AND CONTROL REGISTER	
	BANKSEL	TXSTA
	movlw	B'00100100'
    ;         0-------  CSRC 0=don't care in asynchronous mode
    ;         -0------  TX9 0=Selects 8-bit transmission
    ;         --1-----  TXEN 0=Transmit enabled
    ;         ---0----  SYNC 0=Asynchronous mode
    ;         ----0---  unused
    ;         -----1--  BRGH 1=In asynchronous mode High speed
    ;         ------0-  TRMT 0=TSR full
    ;         -------0  TX9D 0=9th bit of transmit data. Can be PARITY bit	
	movwf	TXSTA

	IF SPBRG_BRGH != 1     ; USART SYNC=0; SPEN=1; CREN=1; SREN=0;
		bcf TXSTA,BRGH
	ENDIF

	; Set up SPRG: Baud Rate Generator Register
	movlw	SPBRG_VALUE 
	movwf	SPBRG
	
	;----------------
	; LOGIN COUNTDOWN
	;----------------
	bcf		STATUS,RP0
	movlw	ELOADER_TIMER1_OVF
	movwf	ELoaderOVFtime
	; Settle TIMER1 CONTROL REGISTER	
	movlw	B'00000000' | ELOADER_T1CKPS ; Prescaler value depending on FOSC
    ;         00------  Unimplemented: Read as '0'
    ;         --00----  T1CKPS1:T1CKPS0: Timer1 Input Clock Prescale Select bits
	;						11 = 1:8 Prescale value	
	;						10 = 1:4 Prescale value	
	;						01 = 1:2 Prescale value	
	;						00 = 1:1 Prescale value	
    ;         ----0---  T1OSCEN: Timer1 Oscillator Enable Control bit
	;						0 = Oscillator is shut off	
    ;         -----0--  NOT T1SYNC: Timer1 External Clock Input Synchronization Control bit
	;					This bit is ignored. Timer1 uses the internal clock when TMR1CS = 0. 	
    ;         ------0-  TMR1CS: Timer1 Clock Source Select bit
	;						0 = Internal clock (FOSC/4) 	
    ;         -------0  TMR1ON: Timer1 On bit
	;						1 = Enables Timer1 
	;						0 = Stops Timer1 	
	movwf	T1CON

	; TMR1 Overflow Interrupt Flag not overflowed 	
	bcf    	PIR1,TMR1IF
	movlw  	high ELOADER_TIMER1
	movwf  	TMR1H		       
	movlw  	low ELOADER_TIMER1
	movwf  	TMR1L		       
	bsf    	T1CON,TMR1ON	; TIMER1 On

	call	ELoaderWaitForLogin
	xorlw	ELOADER_LOGIN
	btfss	STATUS,Z
	goto	ELoaderLogout
	movlw	ELOADER_LOGIN	; Restore Query code

;------------------------------------------------------------
ELoaderLoop 				   
;----------

	; switch case ELoaderQcode
	;
	; case: WRITE INTO FLASH RAM
	movwf  	ELoaderQcode
	xorlw  	ELOADER_WRITE		; Unimplemented
	btfsc  	STATUS,Z
	goto   	ELoaderWrite

	; case: WRITE INTO EEPROM
	xorlw  	ELOADER_WRITE^ELOADER_WRITE_EEPROM
	btfsc  	STATUS,Z
	goto   	ELoaderWrite		   

	; case: PRINT EEPROM
	xorlw  	ELOADER_WRITE_EEPROM^ELOADER_PRINT_EEPROM
	btfsc  	STATUS,Z
	goto   	ELoaderPrintEEPROM	

	; case: LOGIN
	xorlw  	ELOADER_PRINT_EEPROM^ELOADER_LOGIN
	btfsc  	STATUS,Z
	goto   	ELoaderLogin		 

	; case: LOGOUT
	xorlw  	ELOADER_LOGIN^ELOADER_LOGOUT
	btfsc  	STATUS,Z 		    
	goto	ELoaderLogout

ELoaderLoopRead
	call   	ELoaderReadByte	; Wait for a Byte from USART
	goto   	ELoaderLoop

;------------
; LOGOUT
ELoaderLogout
	; Stop TIMER1
	clrf  	T1CON			       
	; Reset USART Registers
	clrf  	RCSTA
	bsf   	STATUS,RP0
	clrf  	TXSTA			     
	bcf   	STATUS,RP0
	clrf  	PIR1
	;---------------------
	; Program User Execute
	goto  	ELoaderExit 
	;---------------------

;------------------------------
; LOGIN acknowledgment
; Send start address of ELoader
; TX : Phhll
ELoaderLogin
	movlw	ELOADER_LOGIN ; LOGIN acknowledgment
	call	ELoaderWriteByte
	movlw	high (ELoaderRoot & 0x7FF)
	call	ELoaderWriteByte
	movlw	low (ELoaderRoot & 0x7FF)

	goto	ELoaderWriteLine

;----------------
; Not implemented
ELoaderWriteFlash
	goto	ELoaderLoopRead

;-------------------------
; PrintEEPROM
; Send 2 bytes from EEPROM
; TX Format: Phhll
ELoaderPrintEEPROM
	; Data Address HIGH
	call	ELoaderReadByte
	movwf	ELoaderAddrH
	; Data Address LOW
	call	ELoaderReadByte
	movwf	ELoaderAddrL
	xorlw  	EEPROM_SIZE+1 ; Is End of EEPROM ?
	btfsc  	STATUS,Z
	goto	ELoaderPrintEEPROM_send ; Yes

	bsf		STATUS,RP0		; Bank1
	movfw	ELoaderAddrL
	movwf	EEADR 
	bsf		EECON1,RD		; read
	movf	EEDATA,W		; data ouput Z=1 if zero

	bcf		STATUS,RP0		; Bank0
	movwf	ELoaderDataL
	clrf	ELoaderDataH	; nothing
	; TX : Paabb
ELoaderPrintEEPROM_send
	movlw	ELOADER_PRINT_EEPROM
	call	ELoaderWriteByte ;<<<<<<<<<<<<<<<<<<
	movfw	ELoaderDataH
	call	ELoaderWriteByte ;<<<<<<<<<<<<<<<<<
	movfw	ELoaderDataL

	goto	ELoaderWriteLine	

;------------------------
; Common Write processing 
ELoaderWrite
	; Data Address HIGH
	call	ELoaderReadByte
	movwf	ELoaderAddrH
	movwf	ELoaderChkSum	;init checksum 
	; Data Address LOW
	call	ELoaderReadByte
	movwf	ELoaderAddrL
	addwf	ELoaderChkSum,f	;+ Checksum

	; Data byte HIGH
	call	ELoaderReadByte
	movwf	ELoaderDataH	; hold data into buffer
	addwf	ELoaderChkSum,f ;+ Checksum

	; Data byte LOW
	call	ELoaderReadByte
	movwf	ELoaderDataL	; hold data into buffer
	addwf	ELoaderChkSum,f ;+ Checksum

	; Checksum
	call	ELoaderReadByte 		
	xorwf	ELoaderChkSum,w	; chksum = w ?
	movlw	ELOADER_CHEKSUM_ERR
	btfss	STATUS,Z
	goto	ELoaderWriteLine; checksum error

	; checksum_ok
	; Split Write Processing: Flash RAM or EEPROM
	movfw	ELoaderQcode
	xorlw	ELOADER_WRITE_EEPROM
	btfsc	STATUS,Z
	goto	ELoaderWriteEEPROM	; write byte into eeprom

	goto	ELoaderWriteFlash

;-----------------
ELoaderWriteEEPROM
	; Write byte into EEPROM @Address
	movfw	ELoaderDataL
	bsf		STATUS,RP0		; Bank1
	movwf	EEDATA			
	movf	ELoaderAddrL,w			
	movwf	EEADR				
	bsf		EECON1,WREN		; Allows write cycles	
	; Write Sequence
	movlw	0x55			
	movwf	EECON2			
	movlw	0xAA			
	movwf	EECON2			
	bsf		EECON1,WR		; Initiates a write cycle	
	
	; Wait for write end
	clrwdt					
	btfsc	EECON1,WR		; is done?
	goto	$-2				; no: loop wait
	bcf		EECON1,WREN		; yes: Inhibits write to the data EEPROM	

	; Write verify
	movf	EEDATA,w
	bsf		EECON1,RD		; read the data written
	xorwf	EEDATA,w
	bcf		STATUS,RP0		; Bank0
	movlw	ELOADER_WRITE_ERR			
	btfss	STATUS,Z		; Data read = data written?
	goto	ELoaderWriteLine; Write error

	; Write OK
	movlw	ELOADER_WRITE_ACK			

	goto	ELoaderWriteLine

;---------------------------------------------------------------------- 	
; Write one byte + new line to USART
; Input: WREG = Byte to send
ELoaderWriteLine
	;goto    ELoaderLoop ;<<<<<<<<<<<<<<<<<<<<<<<<<<
	; Send byte + end of line
	call	ELoaderWriteByte
	movlw	NEW_LINE
	call	ELoaderWriteByte

	goto	ELoaderLoopRead ; Next byte>>>>>>>>>>

;---------------------------------------------------------------------- 	
; Write one Byte to USART 
; Input: WREG = Byte to send
ELoaderWriteByte
	clrwdt
	btfss	PIR1,TXIF		       ; while(!TXIF)
	goto	$-2
	movwf	TXREG			       ; TXREG = octet
	return

;---------------------------------------------------------------------- 	
; Read one Byte from USART 
; Output: WREG = Received Byte
ELoaderReadByte
	clrwdt
	btfss	PIR1,RCIF		       
	goto	$-2
	movf	RCREG,w	; Hold byte from RCREG into WREG
	return

;----------------------------------------------------- 	
; Wait for login from Host 
; Output: WREG = 'L' login or 'X' Execute user program
ELoaderWaitForLogin
	clrwdt
	btfss	PIR1,RCIF		       
	goto	$+3		; Nothing received: continue
	movf	RCREG,w ; Hold received byte RCREG into WREG
	return			; Exit OK >>>>>>>>>>>>>>

	btfss  	PIR1,TMR1IF	       		; TIMER1 overflow?
	goto   	ELoaderWaitForLogin 	; No: loop---------->
	bcf    	T1CON,TMR1ON	      	; Yes: TIMER1 Off
	decfsz 	ELoaderOVFtime,f		
	goto   	ELoaderWaitForLoginOVF	; Reset TIMER1------>
	retlw 	'X'    	; Time out WREG=EXIT >>>>>>>>>>>>>>>

	; Set TIMERH
ELoaderWaitForLoginOVF
	bcf    	PIR1,TMR1IF
	movlw  	high ELOADER_TIMER1
	movwf  	TMR1H		       
	bsf    	T1CON,TMR1ON			; TIMER1 On
	goto	ELoaderWaitForLogin
     
;--------------------------
	ORG 	FLASH_PROG_SIZE
;--------------------------

;---------------------------------------------------------------------- 	
; EEPROM                        
;---------------------------------------------------------------------- 	
	org 	EEPROM_DATA	

;--------------
; MESSAGE BLOCK
;--------------
; Your call sign here
EE_CALLSIGN	; don't remove this label	
EE_BUFFER 	DE 0,"1234567890123456789012345678901234567890123456789"
EE_BUFFER51	DE 0,"1234567890123456789012345678901234567890123456789"

;----------------------------------------
;              Month Day   Year  Vnum
EE_VERSION	DE D'10',D'06',D'15',1,0,1
EE_WPM		DE	SPEED_DEFAULT
EE_UCON		DE	UCON_DEFAULT

;----------------------------------------
	ifdef __ADRES_TRON	
; ADC value report for analyze
; 3x3*2bytes: MSB,LSB
EE_ADRES1to9	DE  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
EE_ADRESH0		DE	0	; MSB
EE_ADRESL0		DE	0	; LSB
	endif
	ifdef __ADRES_TRON1	
EE_ADRESH		DE	0	; MSB
EE_ADRESL		DE	0	; LSB
	endif
;----------------------------------------
;----------------------------------------
;----------------------------------------
;----------------------------------------


	END