﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Resources;
using System.IO.Ports;
using System.Threading;
using System.IO;
using System.Globalization;
using KeyerELoader_10.Properties;


namespace KeyerELoader_10
{
    public enum LogMsgType { Incoming, Outgoing, Normal, Warning, Error, TipMsg };

    public partial class Form1 : Form
    {
 
        private CultureInfo EnglishCulture = new CultureInfo("en-US");
        private CultureInfo FrenchCulture = new CultureInfo("fr-FR");

        // Declare a Resource Manager instance.
        ResourceManager LocRM = new ResourceManager("KeyerELoader_10.FormStrings", typeof(Form1).Assembly);
 
        // Various colors for logging info
        private Color[] LogMsgTypeColor = { Color.Blue, Color.Green, Color.Black, Color.Orange, Color.Red, Color.Purple };

        // Temp holder for whether a key was pressed
        private bool KeyHandled = false;

        // The main control for communicating through the RS-232 port
        private SerialPort comport = new SerialPort();

        private Settings settings = Settings.Default;

        private Queue<byte> recievedData = new Queue<byte>(); 
        private string[] EEPROMArrayHex = new string[16];
        private string[] EEPROMarrayChar = new string[16];
        private byte[] EEPROMbuffer;
        private int EEPROMptr;
        private int EEPROMi;
        private int EEPROMlength = 128;
        private int writeCount;
        private int col = 16;
        private int line = 0;
        private int EEPROMstartAddress = 0;
        private bool ViewRun = false;
        private bool Login = false;
        private byte[] ELoaderStartAddress = new byte[2];

        // communication protocol
        internal const byte LOGIN_TO_DEVICE = (byte)'L'; //TX
        internal const byte LOGIN_TO_DEVICE_ACK = (byte)'L'; //RX
        internal const byte LOGOUT_OF_DEVICE = (byte)'X'; //RX

        internal const byte WRITE_TO_EEPROM = (byte)'E'; //TX
        internal const byte WRITE_TO_EEPROM_ACK = (byte)'K'; //RX
        internal const byte WRITE_TO_EEPROM_ERR = (byte)'!'; //RX
        internal const byte WRITE_TO_EEPROM_CHKSUM_ERR = (byte)'?'; //RX
        internal const byte WRITE_BOUNDARY_ERR = (byte)'$'; //RX

        internal const byte PRINT_EEPROM = (byte)'P'; //RX

        public Form1()
        {

            //Load user settings
            settings.Reload();
            if (settings.LanguageFrench)

                // Définition de la culture par défaut
                Thread.CurrentThread.CurrentUICulture = FrenchCulture;
            else
                Thread.CurrentThread.CurrentUICulture = EnglishCulture;
           
            InitializeComponent();

            // Restore the users settings
            InitializeControlValues();

            // Enable/disable controls based on the current state
            EnableControls();

            // When data is recieved through the port, call this method
            comport.DataReceived += new SerialDataReceivedEventHandler(RXdata);

        }

        public static float OscillatorFrequency
        {
            // Return the value stored in a field.
            get { return Settings.Default.OscillatorFrequency; }
            // Store the value in the field.
            set { Settings.Default.OscillatorFrequency = value; }

        }

        public static decimal BaudError
        {
            // Return the value stored in a field.
            get { return Settings.Default.BaudError; }
            // Store the value in the field.
            set { Settings.Default.BaudError = value; }

        }

        public static bool SaveChangesBaud
        {
            // Return the value stored in a field.
            get { return Settings.Default.SaveChangesBaud; }
            // Store the value in the field.
            set { Settings.Default.SaveChangesBaud = value; }

        }

        public static bool SaveChangesTimeOut
        {
            // Return the value stored in a field.
            get { return Settings.Default.SaveChangesTimeOut; }
            // Store the value in the field.
            set { Settings.Default.SaveChangesTimeOut = value; }

        }

        public static decimal TimeOut
        {
            // Return the value stored in a field.
            get { return Settings.Default.TimeOut; }
            // Store the value in the field.
            set { Settings.Default.TimeOut = value; }

        }
 
        /// <summary> Populate the form's controls with default settings. </summary>
        private void InitializeControlValues()
        {
            //parityComboBox.Items.Clear(); parityComboBox.Items.AddRange(Enum.GetNames(typeof(Parity)));
            parityComboBox.Items.Clear(); 
            parityComboBox.Items.AddRange(GetLocalizedEnum("Parity_", Enum.GetNames(typeof(Parity))));
            stopBitsComboBox.Items.Clear();
            //stopBitsComboBox.Items.AddRange(Enum.GetNames(typeof(StopBits)));
            stopBitsComboBox.Items.AddRange(GetLocalizedEnum("StopBits_", Enum.GetNames(typeof(StopBits))));

            //parityComboBox.Text = settings.Parity.ToString();
            //parityComboBox.Text = GetLocalizedEnum(settings.Parity.ToString(), "Parity_", Enum.GetNames(typeof(Parity)));
            parityComboBox.Text = LocRM.GetString("Parity_"+settings.Parity.ToString());
            //stopBitsComboBox.Text = settings.StopBits.ToString();
            //stopBitsComboBox.Text = GetLocalizedEnum(settings.StopBits.ToString(), "StopBits_", Enum.GetNames(typeof(StopBits)));
            stopBitsComboBox.Text = LocRM.GetString("StopBits_" + settings.StopBits.ToString());
            dataBitsComboBox.Text = settings.DataBits.ToString();
            baudRateComboBox.Text = settings.BaudRate.ToString();
            
            beaconTime0NumericUpDown.Value = settings.BeaconTime0;
            beaconTime1NumericUpDown.Value = settings.BeaconTime1;
            beaconTime2NumericUpDown.Value = settings.BeaconTime2;
            
            msg0TextBox.Text = settings.MsgText0;
            msg1TextBox.Text = settings.MsgText1;
            msg2TextBox.Text = settings.MsgText2;
            
            callSignTextBox.Text = settings.CallSign;
            
            EEPROMlengthNumericUpDown.Value = settings.EEPROMlength;

            msgLength0NumericUpDown.Value = settings.MsgLength0;
            msgLength1NumericUpDown.Value = settings.MsgLength1;
            msgLength2NumericUpDown.Value = settings.MsgLength2;
            MsgId0NumericUpDown.Value = settings.MsgId0;
            MsgId1NumericUpDown.Value = settings.MsgId1;
            MsgId2NumericUpDown.Value = settings.MsgId2;

            timerIntervalNumericUpDown.Value = settings.TimerInterval;
            
            msg0TextBox.MaxLength = Convert.ToInt32(settings.MsgLength0);
            msg1TextBox.MaxLength = Convert.ToInt32(settings.MsgLength1);
            msg2TextBox.MaxLength = Convert.ToInt32(settings.MsgLength2);
            
            comTimer.Interval = Convert.ToInt32(settings.TimerInterval);
            
            langFrRadioButton.Checked = settings.LanguageFrench;
            langEnRadioButton.Checked = settings.LanguageEnglish;
            
            EEPROMstartAddressTextBox.Text = settings.EEPROMstartAddress;
            EEPROMstartAddress = Int32.Parse(settings.EEPROMstartAddress.Trim(), System.Globalization.NumberStyles.HexNumber);
            
            RefreshComPortList();

            // If it is still available, select the last com port used
            if (portNameComboBox.Items.Contains(settings.PortName)) portNameComboBox.Text = settings.PortName;
            else if (portNameComboBox.Items.Count > 0) portNameComboBox.SelectedIndex = portNameComboBox.Items.Count - 1;
            else
            {
                MessageBox.Show(this, LocRM.GetString("NoCOMportDetected"), LocRM.GetString("NoCOMportInstalled"), MessageBoxButtons.OK, MessageBoxIcon.Error);
                this.Close();
            }
        }

        private void RefreshComPortList()
        {
            // List of com port names has changed since last checked?
            string selected = RefreshComPortList(portNameComboBox.Items.Cast<string>(), portNameComboBox.SelectedItem as string, comport.IsOpen);

            // If there was an update, then update the control showing the user the list of port names
            if (!String.IsNullOrEmpty(selected))
            {
                portNameComboBox.Items.Clear();
                portNameComboBox.Items.AddRange(OrderedPortNames());
                portNameComboBox.SelectedItem = selected;
            }
        }

        private string RefreshComPortList(IEnumerable<string> PreviousPortNames, string CurrentSelection, bool PortOpen)
        {
            // Create a new return report to populate
            string selected = null;

            // Retrieve the list of ports currently mounted by the operating system (sorted by name)
            string[] ports = SerialPort.GetPortNames();

            // First determain if there was a change (any additions or removals)
            bool updated = PreviousPortNames.Except(ports).Count() > 0 || ports.Except(PreviousPortNames).Count() > 0;

            // If there was a change, then select an appropriate default port
            if (updated)
            {
                // Use the correctly ordered set of port names
                ports = OrderedPortNames();

                // Find newest port if one or more were added
                string newest = SerialPort.GetPortNames().Except(PreviousPortNames).OrderBy(a => a).LastOrDefault();

                // If the port was already open... (see logic notes and reasoning in Notes.txt)
                if (PortOpen)
                {
                    if (ports.Contains(CurrentSelection)) selected = CurrentSelection;
                    else if (!String.IsNullOrEmpty(newest)) selected = newest;
                    else selected = ports.LastOrDefault();
                }
                else
                {
                    if (!String.IsNullOrEmpty(newest)) selected = newest;
                    else if (ports.Contains(CurrentSelection)) selected = CurrentSelection;
                    else selected = ports.LastOrDefault();
                }
            }

            // If there was a change to the port list, return the recommended default selection
            return selected;
        }

        private string[] OrderedPortNames()
        {
            // Just a placeholder for a successful parsing of a string to an integer
            int num;

            // Order the serial port names in numberic order (if possible)
            return SerialPort.GetPortNames().OrderBy(a => a.Length > 3 && int.TryParse(a.Substring(3), out num) ? num : 0).ToArray();
        }
/*
        /// <summary> Get localized string from ressource. </summary>
        /// <param name="prefix"> prefix = "Parity_" or "StopBits_". </param>
        /// <param name="def"> The default Enum string array. </param>
        /// <param name="oneDef"> The value to find in def. </param>
        private string GetLocalizedEnum(string oneDef, string prefix, string[] def)
        {
            int i = 0;
            string loc = def[0];
            if (def.Contains(oneDef))
            {
                // Do something if the value is available in Array.
                loc = Array.Find(def, s => s.Equals(oneDef));
            }
            /*
            foreach (string str in def)
            {
                if (str.Equals(oneDef))
                {
                    // Example: Get string "Parity_None"
                    // ret = "Aucune" (Français)
                    // ret = "None" (English)
                    loc = LocRM.GetString(prefix + str);
                    break;
                }
                i++;
            }
            return loc;
        }
*/
        /// <summary> Get localized string array from ressource. </summary>
        /// <param name="prefix"> prefix = "Parity_" or "StopBits_". </param>
        /// <param name="def"> The default Enum string array. </param>
        private string[] GetLocalizedEnum(string prefix, string[] def)
        {
            string[] localArray = new string[def.Length];
            def.CopyTo(localArray, 0);
            int i = 0;
            foreach (string str in def)
            {
                localArray[i++] = LocRM.GetString(prefix + str);
            }

            return localArray;
        }


        /// <summary> Get default string from ressource. </summary>
        /// <param name="prefix"> prefix = "Parity_" or "StopBits_". </param>
        /// <param name="def"> The default Enum string array. </param>
        /// <param name="oneLoc"> The value to find in ressource. </param>
        private string GetDefaultEnum(string oneLoc, string prefix, string[] def)
        {
            int i = 0;
            string oneDef = def[0]; // if not found
            foreach (string str in def)
            {
                if (LocRM.GetString(prefix+str).Equals(oneLoc))
                {
                    oneDef = def[i];
                    break;
                }
                i++;
            }

            return oneDef;
        } 

        /// <summary> Save the user's settings. </summary>
        private void SaveSettings()
        {
            settings.BaudRate = int.Parse(baudRateComboBox.Text);
            settings.DataBits = int.Parse(dataBitsComboBox.Text);
            //settings.Parity = (Parity)Enum.Parse(typeof(Parity), parityComboBox.Text);
            settings.Parity = (Parity)Enum.Parse(typeof(Parity), GetDefaultEnum(parityComboBox.Text, "Parity_", Enum.GetNames(typeof(Parity))));
            //settings.StopBits = (StopBits)Enum.Parse(typeof(StopBits), stopBitsComboBox.Text);
            settings.StopBits = (StopBits)Enum.Parse(typeof(StopBits), GetDefaultEnum(stopBitsComboBox.Text, "StopBits_", Enum.GetNames(typeof(StopBits))));
            settings.PortName = portNameComboBox.Text;
            settings.BeaconTime0 = beaconTime0NumericUpDown.Value;
            settings.BeaconTime1 = beaconTime1NumericUpDown.Value;
            settings.BeaconTime2 = beaconTime2NumericUpDown.Value;
            settings.MsgText0 = msg0TextBox.Text;
            settings.MsgText1 = msg1TextBox.Text;
            settings.MsgText2 = msg2TextBox.Text;
            settings.CallSign = callSignTextBox.Text;
            settings.MsgLength0 = msgLength0NumericUpDown.Value;
            settings.MsgLength1 = msgLength1NumericUpDown.Value;
            settings.MsgLength2 = msgLength2NumericUpDown.Value;
            settings.MsgId0 = (byte)MsgId0NumericUpDown.Value;
            settings.MsgId1 = (byte)MsgId1NumericUpDown.Value;
            settings.MsgId2 = (byte)MsgId2NumericUpDown.Value;

            settings.EEPROMlength = EEPROMlengthNumericUpDown.Value;
            settings.TimerInterval = timerIntervalNumericUpDown.Value;
            settings.LanguageFrench = langFrRadioButton.Checked;
            settings.LanguageEnglish = langEnRadioButton.Checked;
            settings.EEPROMstartAddress = EEPROMstartAddressTextBox.Text;

            settings.Save();
        }

        /// <summary> Enable/disable controls based on the app's current state. </summary>
        private void EnableControls()
        {
            msg0TextBox.Enabled = sendMsgButton.Enabled = beaconTime0NumericUpDown.Enabled = comport.IsOpen;
            msg1TextBox.Enabled = beaconTime1NumericUpDown.Enabled = comport.IsOpen;
            msg2TextBox.Enabled = beaconTime2NumericUpDown.Enabled = comport.IsOpen;
            callSignTextBox.Enabled = logoutButton.Enabled = comport.IsOpen;
            EEPROMviewButton.Enabled = loginDeviceButton.Enabled = comport.IsOpen;
            loginToolStripMenuItem.Enabled = comport.IsOpen;
            sendToolStripMenuItem.Enabled = comport.IsOpen;
            logoutToolStripMenuItem.Enabled = comport.IsOpen;
            EEPROMviewToolStripMenuItem.Enabled = comport.IsOpen;
            if (comport.IsOpen)
            {
                openPortButton.Text = LocRM.GetString("ClosePortButton");
                openPortToolStripMenuItem.Text = LocRM.GetString("ClosePortButton");
            }
            else
            {
                openPortButton.Text = LocRM.GetString("OpenPortButton");
                openPortToolStripMenuItem.Text = LocRM.GetString("OpenPortButton");
            }
        }

        private void ClearLog()
        {
            logRichTextBox.Clear();
        }

        /// <summary> Log data to the terminal window. </summary>
        /// <param name="msgtype"> The type of message to be shown. </param>
        /// <param name="msg"> The string containing the message to be shown. </param>
        private void Log(LogMsgType msgtype, string msg)
        {
            logRichTextBox.Invoke(new EventHandler(delegate
            {
                logRichTextBox.SelectedText = string.Empty;
                //logRichTextBox.SelectionFont = new Font(logRichTextBox.SelectionFont, FontStyle.Bold);
                logRichTextBox.SelectionColor = LogMsgTypeColor[(int)msgtype];
                logRichTextBox.AppendText(msg);
                logRichTextBox.ScrollToCaret();
            }));
        }

        /// <summary> Display EEPROM start address. </summary>
        /// <param name="msgtype"> The type of message to be shown. </param>
        /// <param name="s"> The string containing the start address displayed. </param>
        private void ViewEEPROMstartAddress(LogMsgType msgtype, string s)
        {
            EEPROMstartAddressLabel.Invoke(new EventHandler(delegate
            {
                EEPROMstartAddressLabel.Font = new Font(EEPROMstartAddressLabel.Font, FontStyle.Bold);
                EEPROMstartAddressLabel.ForeColor = LogMsgTypeColor[(int)msgtype];
                EEPROMstartAddressLabel.Text = s;
            }));
        }

        /// <summary> Display one line EEPROM. </summary>
        /// <param name="msgtype"> The type of message to be shown. </param>
        /// <param name="lvi"> The line displayed. </param>
        private void ViewEEPROM(LogMsgType msgtype, ListViewItem lvi)
        {
            EEPROMlistView.Invoke(new EventHandler(delegate
            {
                lvi.UseItemStyleForSubItems = false; // for changing color of list view cell
                foreach (ListViewItem.ListViewSubItem si in lvi.SubItems)
                    if (!si.Equals(lvi.SubItems[0]))
                        si.ForeColor = LogMsgTypeColor[(int)msgtype]; // color all columns except first
                EEPROMlistView.Items.Add(lvi);
            }));
        }


        /// <summary> Display ELoader start address. </summary>
        /// <param name="msgtype"> The type of message to be shown. </param>
        /// <param name="s"> The string containing the start address displayed. </param>
        private void ViewELoaderStartAddress(LogMsgType msgtype, string s)
        {
            ELoaderStartAddressLabel.Invoke(new EventHandler(delegate
            {
                ELoaderStartAddressLabel.Font = new Font(ELoaderStartAddressLabel.Font, FontStyle.Bold);
                ELoaderStartAddressLabel.ForeColor = LogMsgTypeColor[(int)msgtype];
                ELoaderStartAddressLabel.Text = s;
            }));
        }

        /// <summary> FIFO Queue. </summary>
        void RXqueueProcess()
        {
            int n = 0;
            //Log(LogMsgType.Incoming, "ProcessData count" + ": " + recievedData.Count + "\n");
 
            // Determine if we have a Frame in the queue
            // Parse Received Frame Signature
            if ((recievedData.First() == 0) & (recievedData.Count == 1))
                n = 1;
            else
            if ((recievedData.First() == LOGIN_TO_DEVICE_ACK) & (recievedData.Count == 4))
                n = 4;
            else
            if ((recievedData.First() == WRITE_TO_EEPROM_ACK) & (recievedData.Count == 2))
                n=2;
            else
            if ((recievedData.First() == WRITE_BOUNDARY_ERR) & (recievedData.Count == 2))
                n=2;
            else
            if ((recievedData.First() == WRITE_TO_EEPROM_CHKSUM_ERR) & (recievedData.Count == 2))
                n=2;
            else
            if ((recievedData.First() == WRITE_TO_EEPROM_ERR) & (recievedData.Count == 2))
                n=2;
            else
            if ((recievedData.First() == PRINT_EEPROM) & (recievedData.Count == 4))
                n=4;
            //Log(LogMsgType.Incoming, "ProcessData n" + ": " + n + "\n");
            
            if (n!=0)
            {
                byte[] data = new byte[n];
                recievedData.CopyTo(data, 0);
                //Enumerable.Range(0, n).Select(i => recievedData.Dequeue());
                recievedData.Clear();
                RXframeProcess(data);
                //Log(LogMsgType.Incoming, "ProcessData" + ": " + BitConverter.ToString(data) + "\n");
                return;
            }
        }

        /// <summary> Data received into port buffer queue. </summary>
        private void RXdata(object sender, SerialDataReceivedEventArgs e)
        {
            // If the com port has been closed, do nothing
            if (!comport.IsOpen) return;

            //---------------------------------------------------------------
            // PIC send all messages ended by 0x0A
            // Input format:
            // "L"aabb0A = Login acknowledgment: aabb = ELoader start address
            // "K"0A = Write EEPROM acknowledgment
            // "P"aabb0A = Print EEPROM byte
            //---------------------------------------------------------------
            byte[] data = new byte[comport.BytesToRead];
            comport.Read(data, 0, data.Length);
            data.ToList().ForEach(b => recievedData.Enqueue(b));
            //Log(LogMsgType.Incoming, "RXData" + ": " + BitConverter.ToString(data) + "\n");
            RXqueueProcess();
        }

        /// <summary> RX Byte Frame process. </summary>
        /// <param name="data"> The Byte Frame to process. </param>
        void RXframeProcess(byte[] data)
         {
           if (!Login)
            {
                if ((data.Length > 3) & (data[0] == LOGIN_TO_DEVICE_ACK))
                {
                    Login = true;
                    Log(LogMsgType.Incoming, LocRM.GetString("Readytoload")+": " + BitConverter.ToString(data) + "\n");
                    // Store & display ELoader and EEPROM start address
                    ELoaderStartAddress[0] = data[1];
                    ELoaderStartAddress[1] = data[2];
                    ViewELoaderStartAddress(LogMsgType.Incoming, BitConverter.ToString(ELoaderStartAddress).Replace("-", string.Empty));
                    ViewEEPROMstartAddress(LogMsgType.Incoming, settings.EEPROMstartAddress.Trim());
                }
            }
            else
            {
                if (data.Length > 1)
                {
                    //byte c = data[0];
                    switch (data[0])
                    {
                        case WRITE_TO_EEPROM_ACK:
                            // Write next byte
                            Log(LogMsgType.Incoming, LocRM.GetString("WritetoEEPROMACK") + ": " + BitConverter.ToString(data) + "\n");
                            writeCount = 2;
                            EEPROMi++;
                            if (EEPROMi < EEPROMbuffer.Length)
                                TXdata();
                            else
                            {
                                Log(LogMsgType.Incoming, LocRM.GetString("WritetoEEPROMCompleted") + "\n");
                                // EEPROM dump
                                EEPROMviewButton_Click(this, new EventArgs());
                            }
                            break;
                        case WRITE_BOUNDARY_ERR:
                        case WRITE_TO_EEPROM_CHKSUM_ERR:
                        case WRITE_TO_EEPROM_ERR:
                            Log(LogMsgType.Incoming, LocRM.GetString("WritetoEEPROMfailed") + ": " + BitConverter.ToString(data) + "\n");
                            writeCount--;
                            // Resend byte
                            if (writeCount > 0) TXdata();
                            break;

                         case PRINT_EEPROM: // Paabb
                            Log(LogMsgType.Incoming, LocRM.GetString("DumpEEPROM") + ": " + BitConverter.ToString(data) + "\n");

                            // Hold printable ASCII char 33 to 127, other replace by a dot
                            if (data[2] > 31 && data[2] < 128)
                                EEPROMarrayChar[16 - col] = ((char)data[2]).ToString();
                            else
                                EEPROMarrayChar[16 - col] = ".";

                            // Hold hex value    
                            byte[] oneEEPROMbyte = { data[2] };
                            EEPROMArrayHex[16 - col] = BitConverter.ToString(oneEEPROMbyte);

                            col = col - 1;
                            // Print line of 16 bytes
                            if (col < 1)
                            {
                                line = line + 10;
                                // 1st col=: Address
                                ListViewItem lvi = new ListViewItem(string.Format("{0,2:D2}", line));
                                // next 16 col: hex value
                                foreach (string s in EEPROMArrayHex)
                                    lvi.SubItems.Add(s);
                                // last col: string of char
                                lvi.SubItems.Add(string.Join("", EEPROMarrayChar));
                                // Add new line
                                ViewEEPROM(LogMsgType.Incoming, lvi);

                                col = 16; // for next line
                            }

                            // Query next data
                            // Query = P2100 + i
                            EEPROMi++;
                            if (EEPROMi > EEPROMlength - 1)
                            {
                                Log(LogMsgType.Incoming, LocRM.GetString("DumpEEPROMCompleted") + "\n");
                                ViewRun = false;
                            }
                            else
                            {
                                byte[] ptr = BitConverter.GetBytes(EEPROMptr + EEPROMi);
                                byte[] query = { (byte)PRINT_EEPROM, ptr[1], ptr[0] };
                                TXquery(query, LocRM.GetString("QueryDumpEEPROMNext") + ": "); // Next EEPROM Dump
                                //ViewRun = false;
                            }

                            break;
                        default:
                            Log(LogMsgType.Incoming, "?: " + BitConverter.ToString(data) + "\n");
                            break;
                    }
                }
            }

        }

        private int GetEEPROMptr()
        {
            //return (int)0x2100;
            return EEPROMstartAddress;
        }

        /// <summary> Get EEPROM buffer</summary>
        /// <param name="callSign"> The Call Sign to send. </param>
        /// <param msgn="msg0"> The message n to send. </param>
        /// <param msgn="bTime0"> The message n Beacom Time to send. </param>
        private void GetEEPROMbuffer(TextBox callSign, TextBox msg0, TextBox msg1, TextBox msg2, byte bTime0, byte bTime1, byte bTime2)
        {
            EEPROMptr = GetEEPROMptr();
            EEPROMi = 0;
            byte[] buffer = BitConverter.GetBytes(EEPROMptr);
            EEPROMbuffer = System.Text.ASCIIEncoding.Default.GetBytes(callSign.Text.ToUpper() + "0" +
                "000" + msg0.Text.ToUpper() + "0" +
                "000" + msg1.Text.ToUpper() + "0" +
                "000" + msg2.Text.ToUpper() + "0");
            Log(LogMsgType.Outgoing, LocRM.GetString("EEPROMbufferlength")+": " + EEPROMbuffer.Length + "\n");
            
            int i = callSign.Text.Length;
            EEPROMbuffer[i] = 0x00; //end of message
            EEPROMbuffer[i + 1] = 0x02; //head of message
            EEPROMbuffer[i + 2] = settings.MsgId0; //Id keypad message
            EEPROMbuffer[i + 3] = bTime0; //Beacon Time message
            
            i += msg0.Text.Length + 1 + 3;
            EEPROMbuffer[i] = 0x00; //end of message
            EEPROMbuffer[i + 1] = 0x02; //head of message
            EEPROMbuffer[i + 2] = settings.MsgId1; //Id keypad message
            EEPROMbuffer[i + 3] = bTime1; //Beacon Time message
            
            i += msg1.Text.Length + 1 + 3;
            EEPROMbuffer[i] = 0x00; //end of message
            EEPROMbuffer[i + 1] = 0x02; //head of message
            EEPROMbuffer[i + 2] = settings.MsgId2; //Id keypad message
            EEPROMbuffer[i + 3] = bTime2; //Beacon Time message
            
            EEPROMbuffer[EEPROMbuffer.Length - 1] = 0x00; //end of message
        }

        /// <summary> Transmit the Query EEPROM data.</summary>
        private void TXdata()
        {
            //EEPROMptr = EEPROMptr + EEPROMi;
            byte[] EEPROMaddr = BitConverter.GetBytes(EEPROMptr + EEPROMi);
            byte[] buffer = { 0x00, EEPROMaddr[1], EEPROMaddr[0], 0x00, EEPROMbuffer[EEPROMi], 0x00 };

            int checksum = 0;
            foreach (byte b in buffer) checksum += b;
            checksum &= 0xff;
            
            buffer[buffer.Length - 1] = (byte)checksum;
            buffer[0] = (byte)WRITE_TO_EEPROM;
            
            Log(LogMsgType.Outgoing, LocRM.GetString("SendEEPROMbyte")+": " +  BitConverter.ToString(buffer) + "\n");
            // Send the binary data via the port
            comport.Write(buffer, 0, buffer.Length);
        }

        /// <summary> Transmit the Query data.</summary>
        /// <param name="queryId"> The Query Code Id. </param>
        /// <param name="logText"> The Text to show to the Log. </param>
        private void TXquery(byte queryId, string logText)
        {
            try
            {
                byte[] buffer = System.Text.ASCIIEncoding.Default.GetBytes("0");
                buffer[0] = queryId;
                // Display the hex digits on the log
                Log(LogMsgType.Outgoing, logText + (char)queryId + "\n");

                // Send the binary data out the port
                comport.Write(buffer, 0, buffer.Length);
            }
            catch (FormatException)
            {
                // Hex string error
                Log(LogMsgType.Error, LocRM.GetString("Wrongformathexstring")+": " + queryId + "\n");
            }
        }

        /// <summary> Transmit the Query data.</summary>
        /// <param name="queryId"> The Query Code Id. </param>
        /// <param name="logText"> The Text to show to the Log. </param>
        private void TXquery(byte[] query, string logText)
        {
            try
            {
                // Display the hex digits on the log
                Log(LogMsgType.Outgoing, logText + BitConverter.ToString(query) + "\n");
                // Send the binary data out the port
                comport.Write(query, 0, query.Length);
            }
            catch (FormatException)
            {
                // Hex string error
                Log(LogMsgType.Error, LocRM.GetString("Wrongformathexstring")+": " + query + "\n");
            }
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            // Create the ToolTip and associate with the Form container.
            ToolTip toolTip1 = new ToolTip();

            // Force the ToolTip text to be displayed whether or not the form is active.
            toolTip1.ShowAlways = true;

            // Set up the ToolTip text for the Button and Checkbox.
            toolTip1.SetToolTip(this.saveSettingButton, LocRM.GetString("Pleasecloseandrestartaftersaving"));
            
            ClearLog();
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            // disconnect device
            if (Login) logoutButton_Click(sender, e);

            // The form is closing, save the user's preferences
            SaveSettings();
        }

        private void baudRateComboBox_Validating(object sender, CancelEventArgs e)
        {
            int x; e.Cancel = !int.TryParse(baudRateComboBox.Text, out x);
        }

        private void dataBitsComboBox_Validating(object sender, CancelEventArgs e)
        {
            int x; e.Cancel = !int.TryParse(dataBitsComboBox.Text, out x);
        }

        private void Form1_Shown(object sender, EventArgs e)
        {
            //clearLogButton_Click(sender, e);
            Log(LogMsgType.Normal, String.Format(LocRM.GetString("ApplicationStartedAt") + " {0}\n", DateTime.Now));
            //Log(LogMsgType.TipMsg, String.Format("Timer Interval:" + " {0,2} ms\n", comTimer.Interval));
            //Log(LogMsgType.TipMsg, String.Format("Message0 Length:" + " {0,2}\n", msg0TextBox.MaxLength));
            //Log(LogMsgType.TipMsg, String.Format("Message1 Length:" + " {0,2}\n", msg1TextBox.MaxLength));
            //Log(LogMsgType.TipMsg, String.Format("Message2 Length:" + " {0,2}", msg2TextBox.MaxLength)+"\n");
            //Log(LogMsgType.TipMsg, String.Format("EEPROM Buffer Message:" + " {0,2}", EEPROMlengthNumericUpDown.Value) + "\n");
            //Log(LogMsgType.TipMsg, String.Format("EEPROM Start Address:" + " {0,6}", EEPROMstartAddress) + "\n");
            openPortButton.Focus();
        }

        private void openPortButton_Click(object sender, EventArgs e)
        {
            bool error = false;

            // If the port is open, close it.
            if (comport.IsOpen)
            {
                comport.Close();
                Log(LogMsgType.Normal, LocRM.GetString("PortClosed") + "\n");
            }
            else
            {
                // Set the port's settings
                comport.BaudRate = int.Parse(baudRateComboBox.Text);
                comport.DataBits = int.Parse(dataBitsComboBox.Text);
                comport.StopBits = (StopBits)Enum.Parse(typeof(StopBits), GetDefaultEnum(stopBitsComboBox.Text, "StopBits_", Enum.GetNames(typeof(StopBits))));
                comport.Parity = (Parity)Enum.Parse(typeof(Parity), GetDefaultEnum(parityComboBox.Text, "Parity_", Enum.GetNames(typeof(Parity))));
                comport.PortName = portNameComboBox.Text;

                try
                {
                    // Open the port
                    comport.Open();
                }
                catch (UnauthorizedAccessException) { error = true; }
                catch (IOException) { error = true; }
                catch (ArgumentException) { error = true; }

                if (error) MessageBox.Show(this, LocRM.GetString("CouldNotOpen"), LocRM.GetString("COMportUnavailable"), MessageBoxButtons.OK, MessageBoxIcon.Stop);
                else
                {
                    // message to log
                    Log(LogMsgType.Normal, LocRM.GetString("PortOpened") + "\n");
                }
            }
            // Change the state of the form's controls
            EnableControls();

            // If the port is open, send focus to the 1st message text box
            if (comport.IsOpen)
            {
                msg0TextBox.Focus();
            }

        }

        private void clearLogButton_Click(object sender, EventArgs e)
        {
            ClearLog();
        }

        private void msg2TextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            { e.Handled = KeyHandled; }
        }

        private void msg2TextBox_KeyDown(object sender, KeyEventArgs e)
        {
            // If the user presses [ENTER], send the data now
            if (KeyHandled = e.KeyCode == Keys.Enter)
            { e.Handled = true; sendMsgButton_Click(sender, e); }

        }

        private void sendMsgButton_Click(object sender, EventArgs e)
        {
            // If it is still available, select the last com port used
            if (msg0TextBox.Text.Length + msg1TextBox.Text.Length + msg0TextBox.Text.Length > EEPROMlengthNumericUpDown.Value)
                MessageBox.Show(this, LocRM.GetString("MessageslengthGTEEPROMbuffer")+": " + string.Format(" {0,2}\n", EEPROMlengthNumericUpDown.Value), "EEPROM buffer Boundary Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            else
            {
                EEPROMlistView.Items.Clear();
                GetEEPROMbuffer(callSignTextBox, msg0TextBox, msg1TextBox, msg2TextBox,
                    (byte)beaconTime0NumericUpDown.Value, 
                    (byte)beaconTime1NumericUpDown.Value, 
                    (byte)beaconTime2NumericUpDown.Value);
                TXdata();
                msg0TextBox.SelectAll();
                msg0TextBox.Focus();
            }

        }

        private void loginDeviceButton_Click(object sender, EventArgs e)
        {
            TXquery((byte)LOGIN_TO_DEVICE, LocRM.GetString("Login")+": ");
        }

        private void sendNextButton_Click(object sender, EventArgs e)
        {
            EEPROMi++;
            if (EEPROMi < EEPROMbuffer.Length)
            {
                TXdata();
            }
        }

        private void EEPROMviewButton_Click(object sender, EventArgs e)
        {
            if (!ViewRun) 
            {
                ViewRun = true;
                EEPROMlistView.Items.Clear(); //Clear view
                col = 16;
                line = -10;
                EEPROMptr = GetEEPROMptr();
                EEPROMi = 0;
                byte[] ptr = BitConverter.GetBytes(EEPROMptr);
                // Query = P2100
                byte[] query = { (byte)PRINT_EEPROM, ptr[1], ptr[0] };
                TXquery(query, LocRM.GetString("QueryDumpEEPROMStart")+": "); // Start EEPROM Dump
             }
        }

        private void saveSettingButton_Click(object sender, EventArgs e)
        {
            SaveSettings();
         }

        private void EEPROMstartAddressTextBox_Validating(object sender, CancelEventArgs e)
        {
            foreach (char c in EEPROMstartAddressTextBox.Text.Trim())
            {
                // Check for numeric characters (hex in this case).
                if (!((c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f')))
                {
                    MessageBox.Show(this, LocRM.GetString("WrongHexformat") + string.Format(" {0} .\n", EEPROMstartAddressTextBox.Text), "EEPROM Start Address Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    EEPROMstartAddressTextBox.SelectAll();
                    EEPROMstartAddressTextBox.Focus();
                    break;
                }
            }
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Show the user the about dialog
            (new AboutForm()).ShowDialog(this);
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void openPortToolStripMenuItem_Click(object sender, EventArgs e)
        {
            openPortButton_Click(sender, e);
        }

        private void loginToolStripMenuItem_Click(object sender, EventArgs e)
        {
            loginDeviceButton_Click(sender, e);
        }

        private void sendToolStripMenuItem_Click(object sender, EventArgs e)
        {
            sendMsgButton_Click(sender, e);
        }

        private void logoutButton_Click(object sender, EventArgs e)
        {
            TXquery((byte)LOGOUT_OF_DEVICE, LocRM.GetString("Logout")+": ");
            Login = false;
        }

        private void logoutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            logoutButton_Click(sender, e);
        }

        private void EEPROMstartAddressTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            char keyChar;
            keyChar = e.KeyChar;

            if (!(Char.IsDigit(keyChar) // 0 - 9 A B C D E F
               || keyChar == 'A' || keyChar == 'B' || keyChar == 'C'
               || keyChar == 'D' || keyChar == 'E' || keyChar == 'F'
               || keyChar == 8  // backspace
               || keyChar == 13  // enter
               ))
            {
                //  Do not display the keystroke
                e.Handled = true;
            }
        }

        private void clearLogToolStripMenuItem_Click(object sender, EventArgs e)
        {
            clearLogButton_Click(sender, e);
        }

        private void viewEEPROMtoolStripMenuItem_Click(object sender, EventArgs e)
        {
            EEPROMviewButton_Click(sender, e);
        }

        private void baudRateCalculatorToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Show the user the about dialog
            (new BaudRateCalculForm()).ShowDialog(this);
        }

        private void timeOutCalculatorToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Show the user the about dialog
            (new TimeOutCalculForm()).ShowDialog(this);

        }

        private void comTimer_Tick(object sender, EventArgs e)
        {
            // checks to see if COM ports have been added or removed
            // since it is quite common now with USB-to-Serial adapters
            RefreshComPortList();
        }

    }
}
